/*
 * $Id:$
 */

package csbase.client.util.filechooser;

import java.awt.Window;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import csbase.client.util.filechooser.filters.ClientFileAllFilter;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.client.util.filechooser.filters.ClientFileMultipleTypesFilter;
import csbase.client.util.filechooser.filters.ClientFileSingleTypeFilter;
import csbase.logic.ClientFile;
import csbase.logic.ClientSGAFile;

/**
 * Utilitrio para busca de arquivos que esto no SGA.
 * 
 * @author Tecgraf
 */
public class ClientSGAFileChooserUtil {

  /**
   * Mtodo utilitrio usado na escolha de arquivos do SGA com base em lista de
   * tipos aceitveis.
   * 
   * @param window - a janela-me (owner).
 * @param fileTypes - tipos dos arquivo usados no filtro.
 * @param title - ttulo.
 * @param allowAll - determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
 * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  static public ClientSGAFile browseSingleFileInOpenMode(Window window,
    Collection<String> fileTypes, String title, boolean allowAll,
    ClientSGAFile currentDirectory) {
    return browseSingleInOpenMode(window, title, fileTypes,
      currentDirectory, allowAll, ClientFileChooserSelectionMode.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrio localizado no SGA.
   * 
   * @param window - a janela-me (owner).
 * @param fileTypes - tipos dos arquivo usados no filtro.
 * @param title - ttulo.
 * @param allowAll - determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
 * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  static public ClientSGAFile browseSingleDirectoryInOpenMode(Window window,
    Collection<String> fileTypes, String title, boolean allowAll,
    ClientSGAFile currentDirectory) {
    return browseSingleInOpenMode(window, title, fileTypes,
      currentDirectory, allowAll,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrios localizados no
   * SGA.
   * 
   * @param window - a janela-me (owner).
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param title - ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos". 
   * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  static public List<ClientSGAFile> browseMultipleFilesInOpenMode(
    Window window, Collection<String> fileTypes, String title,
    boolean allowAll, ClientSGAFile currentDirectory) {
    return browseMultipleInOpenMode(window, title, fileTypes, allowAll,
      currentDirectory, ClientFileChooserSelectionMode.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrios localizados no
   * SGA.
   * 
   * @param window - a janela-me (owner).
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param title - ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos". 
   * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  static public List<ClientSGAFile> browseMultipleDirectoriesInOpenMode(
    Window window, Collection<String> fileTypes, String title,
    boolean allowAll, ClientSGAFile currentDirectory) {
    return browseMultipleInOpenMode(window, title, fileTypes, allowAll,
      currentDirectory, ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos do SGA com base em
   * lista de tipos aceitveis.
   * 
   * @param window - a janela-me (owner).
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param defaultFileType tipo default.
   * @param title - ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @return o arquivo escolhido ou {@code null}.
   */
  static public ClientSGAFile browseSingleFileInSaveMode(Window window,
    Collection<String> fileTypes, String defaultFileType, String title,
    boolean allowAll, ClientSGAFile currentDirectory) {
    return browseSingleInSaveMode(window, title, fileTypes, defaultFileType,
      allowAll, currentDirectory, ClientFileChooserSelectionMode.FILES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrio do SGA com base em
   * lista de tipos aceitveis.
   * 
   * @param window - a janela-me (owner).
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param title - ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @return o diretrio escolhido ou {@code null}.
   */
  static public ClientSGAFile browseSingleDirectoryInSaveMode(Window window,
    Collection<String> fileTypes, String defaultFileType, String title,
    boolean allowAll, ClientSGAFile currentDirectory) {
    return browseSingleInSaveMode(window, title, fileTypes, defaultFileType,
      allowAll, currentDirectory,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos do SGA com base em
   * lista de tipos aceitveis.
   * 
   * @param window - a janela-me (owner).
   * @param title - ttulo.
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @param allowAll - determina se a opo de escolher qualquer tipo de arquivo
   *        deve ser habilitada.
   * @param selectionMode - modo de seleo (arquivo ou diretrio).
   * 
   * @return o arquivo escolhido ou {@code null}.
   */
  private static ClientSGAFile browseSingleInOpenMode(Window window,
    String title, Collection<String> fileTypes, ClientSGAFile currentDirectory,
    boolean allowAll, ClientFileChooserSelectionMode selectionMode) {

    ClientFileChooser chooser = new ClientFileChooser(window);
    chooser.setCurrentDirectory(currentDirectory);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.OPEN);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(false);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();

    if (fileTypes != null) {
      if (fileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(fileTypes));
      }
      for (final String fileType : fileTypes) {
        final ClientFileSingleTypeFilter flt =
          new ClientFileSingleTypeFilter(fileType);
        filters.add(flt);
      }
    }

    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);

    if (!filters.isEmpty()) {
        chooser.setSelectedFilter(filters.get(0));
    }

    chooser.setVisible(true);

    List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null || selected.size() == 0) {
      return null;
    }

    ClientSGAFile first = (ClientSGAFile) selected.get(0);
    return first;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos do SGA com base em
   * lista de tipos aceitveis.
   * 
   * @param window - a janela-me (owner).
   * @param title - ttulo.
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos". 
   * @param currentDirectory - o diretrio inicial da janela de escolha.
   * @param selectionMode - modo de seleo (arquivo ou diretrio).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static List<ClientSGAFile> browseMultipleInOpenMode(
    Window window, String title, Collection<String> fileTypes,
    boolean allowAll, ClientSGAFile currentDirectory,
    ClientFileChooserSelectionMode selectionMode) {

    ClientFileChooser chooser = new ClientFileChooser(window);
    chooser.setCurrentDirectory(currentDirectory);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.OPEN);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(false);
    chooser.setCardinality(ClientFileChooserCardinality.MULTIPLE_CHOOSE);

    List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();

    if (fileTypes != null) {
      if (fileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(fileTypes));
      }
      for (final String fileType : fileTypes) {
        final ClientFileSingleTypeFilter flt =
          new ClientFileSingleTypeFilter(fileType);
        filters.add(flt);
      }
    }

    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }

    chooser.setViewFilters(filters);

    if (!filters.isEmpty()) {
        chooser.setSelectedFilter(filters.get(0));
    }

    chooser.setVisible(true);

    List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null) {
      return null;
    }

    List<ClientSGAFile> sgaFileSelected = new ArrayList<ClientSGAFile>();
    for (ClientFile clientFile : selected) {
      ClientSGAFile clientLocalFile = (ClientSGAFile) clientFile;
      sgaFileSelected.add(clientLocalFile);
    }
    return sgaFileSelected;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos do SGA com base em
   * lista de tipos aceitveis.
   * 
   * @param window - a janela-me (owner).
   * @param title - ttulo.
   * @param fileTypes - tipos dos arquivo usados no filtro.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos". 
   * @param currentDirectory - o diretrio inicial da escolha.
   * @param selectionMode - modo de seleo (arquivo ou diretrio).
   * @return o arquivo escolhido ou {@code null}.
   */
  private static ClientSGAFile browseSingleInSaveMode(Window window,
    String title, Collection<String> fileTypes, String defaultFileType,
    boolean allowAll, ClientSGAFile currentDirectory,
    ClientFileChooserSelectionMode selectionMode) {

    ClientFileChooser chooser = new ClientFileChooser(window);
    chooser.setCurrentDirectory(currentDirectory);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(ClientFileChooserType.SAVE);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(false);
    chooser.setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    List<ClientFileFilterInterface> filters =
          new ArrayList<ClientFileFilterInterface>();
    List<String> allFileTypes = new ArrayList<String>();
    if (defaultFileType != null) {
      allFileTypes.add(defaultFileType);
    }
    for (String fileType : fileTypes) {
      if (!allFileTypes.contains(fileType)) {
        allFileTypes.add(fileType);
      }
    }
    ClientFileFilterInterface defaultFilter = null;
    for (final String fileType : allFileTypes) {
      ClientFileSingleTypeFilter filter =
        new ClientFileSingleTypeFilter(fileType);
      filters.add(filter);
      if (defaultFileType != null && defaultFileType.equals(fileType)) {
        defaultFilter = filter;
      }
    }
    if (allFileTypes.size() > 1) {
      filters.add(new ClientFileMultipleTypesFilter(allFileTypes));
    }
    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }
    chooser.setViewFilters(filters);
    if (defaultFilter != null) {
      chooser.setSelectedFilter(defaultFilter);
    }
    chooser.setVisible(true);
    String[] selectedSavePath = chooser.getSelectedSavePath();
    if (selectedSavePath == null || selectedSavePath.length <= 0) {
      return null;
    }

    String path =
      buildAbsolutePath(selectedSavePath, currentDirectory.getSeparator());

    ClientSGAFile clientSGAFile =
      new ClientSGAFile(currentDirectory.getSGAName(), path);

    if (selectionMode == ClientFileChooserSelectionMode.DIRECTORIES_ONLY) {
      clientSGAFile.setDir(true);
    }

    return clientSGAFile;
  }

  /**
   * Constri o path absoluto.
   * 
   * @param path path.
   * @param separator separador.
   * @return path absoluto.
   */
  private static String buildAbsolutePath(String[] path, String separator) {
    StringBuilder absolutePath = new StringBuilder();
    for (String p : path) {
      if (!p.trim().isEmpty()) {
        absolutePath.append(separator);
        absolutePath.append(p);
      }
    }
    return absolutePath.toString();
  }
}
