/*
 * $Id$
 */
package csbase.client.util.gui.log;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import csbase.client.util.event.EventListener;
import csbase.client.util.event.EventManager;
import csbase.client.util.event.IEvent;

/**
 * Controlador de recargas do arquivo corrente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class LogPanelReloader {

  /**
   * Tempo de recarga padro do arquivo de log.
   */
  public static final long DEFAULT_TIME_SECS = 10;
  /**
   * Menor tempo de recarga aceito, em segundos.
   */
  public static final long MIN_TIME_SECS = 5;

  /**
   * Painel de log
   */
  private final Reloadable reloadable;

  /**
   * Indicativo que o reloader est rodando
   */
  private final AtomicBoolean running;

  /**
   * Perodo de reload
   */
  private final AtomicLong period;

  /**
   * Gerente de eventos
   */
  private final EventManager eventManager;

  /**
   * Runnable que efetivamente faz a atualizao do texto.
   */
  private final ReloadTask task;

  /**
   * Thread sobre o qual a tarefa (<code>ReloadTask</code>)  efetivamente
   * rodada.
   */
  private volatile Thread thread;

  /**
   * @param reloadable componente que vai ser recarregado por esta instncia.
   */
  public LogPanelReloader(final Reloadable reloadable) {
    super();
    this.reloadable = reloadable;
    this.period = new AtomicLong();
    this.running = new AtomicBoolean(false);
    this.eventManager = new EventManager();
    this.task = new ReloadTask();
    setTime(DEFAULT_TIME_SECS);
  }

  /**
   * Inicializa a auto-recarga do arquivo de log.
   */
  public final void start() {
    if (running.compareAndSet(false, true) || null == thread
      || !thread.isAlive()) {
      this.thread = new Thread(this.task);
      this.thread.setDaemon(true);
      this.thread.start();
      eventManager.fireEvent(new StatusChangedEvent(true));
    }
    eventManager.fireEvent(new StartEvent());
  }

  /**
   * Para a auto-recarga do arquivo de log.
   */
  public final void stop() {
    if (running.compareAndSet(true, false)) {
      this.thread = null;
      eventManager.fireEvent(new StatusChangedEvent(false));
    }
    eventManager.fireEvent(new StopEvent());
  }

  /**
   * @return true se a auto-recarga estiver ligada.
   */
  public final boolean isRunning() {
    return running.get();
  }

  /**
   * Altera o perodo de tempo para a recarga do arquivo de log.
   * 
   * @param period novo tempo entre cada auto-recarga, em segundos.
   */
  public void setTime(long period) {
    this.period.set(Math.max(MIN_TIME_SECS, period));
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link StatusChangedEvent}.
   * 
   * @param listener o observador.
   */
  public void addStatusChangedEventListener(
    EventListener<StatusChangedEvent> listener) {
    eventManager.addEventListener(listener, StatusChangedEvent.class);
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link StopEvent}.
   * 
   * @param listener o observador.
   */
  public void addStopEventListener(EventListener<StopEvent> listener) {
    eventManager.addEventListener(listener, StopEvent.class);
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link StartEvent}.
   * 
   * @param listener o observador.
   */
  public void addStartEventListener(EventListener<StartEvent> listener) {
    eventManager.addEventListener(listener, StartEvent.class);
  }

  /**
   * Representa um evento de mudana de estado do recarregado que pode estar
   * ligado ou desligado.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public static class StatusChangedEvent implements IEvent {

    /**
     * Indicativo de "em execuo"
     */
    private final boolean running;

    /**
     * Construtor
     * 
     * @param running indicativo de "em execuo"
     */
    StatusChangedEvent(boolean running) {
      this.running = running;
    }

    /**
     * @return true se a auto-recarga foi ligada ao gerar esse evento.
     */
    public boolean isRunning() {
      return running;
    }
  }

  /**
   * Representa um evento de stop no reccarregador.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public static class StopEvent implements IEvent {
    //Subclasse de marcao apenas
  }

  /**
   * Representa um evento de start no reccarregador.
   * 
   * @author Tecgraf / PUC-Rio
   */
  public static class StartEvent implements IEvent {
    //Subclasse de marcao apenas
  }

  /**
   * Recarrega o arquivo de log e dorme pelo perodo definido at a prxima
   * recarga.
   * 
   * @author Tecgraf / PUC-Rio
   */
  class ReloadTask implements Runnable {
    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      Thread thisThread = Thread.currentThread();
      while (thread == thisThread) {
        reloadable.reload();
        try {
          TimeUnit.SECONDS.sleep(period.get());
        }
        catch (InterruptedException e) {
          // No faz nada
        }
      }
    }
  }
}
