package csbase.client.util.table;

import java.awt.Color;
import java.awt.Component;
import java.lang.reflect.Constructor;

import javax.swing.DefaultCellEditor;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.border.LineBorder;
import javax.swing.table.TableCellEditor;

/**
 * {@link TableCellEditor} genrico.
 * 
 * @author Tecgraf
 */
public class GenericCellEditor extends DefaultCellEditor {

  /**
   * Construtor do valor da ltima clula para a qual se criou o componente de
   * edio.
   */
  private Constructor<?> constructor;
  /** Valor da ltima clula editada. */
  private Object value;
  /** Tabela que contm clula para a qual se criou o componente de edio. */
  private JTable table;
  /** Coluna da ltima clula para a qual se criou o componente de edio. */
  private int column;
  /** Linha da ltima clula para a qual se criou o componente de edio. */
  private int row;

  /**
   * Construtor.
   * 
   * @param textField campo de texto a ser utilizado na edio.
   */
  public GenericCellEditor(JTextField textField) {
    super(textField);
    getComponent().setName("Table.editor");
  }

  /**
   * Construtor.
   */
  public GenericCellEditor() {
    this(new JTextField());
  }

  /**
   * Define o alinhamento horizontal do texto na clula.
   * 
   * @param alignment o alinhamento horizontal do texto.<br>
   *        Os valroes vlidos so:
   *        <ul>
   *        <li><code>JTextField.LEFT</code>
   *        <li><code>JTextField.CENTER</code>
   *        <li><code>JTextField.RIGHT</code>
   *        <li><code>JTextField.LEADING</code>
   *        <li><code>JTextField.TRAILING</code>
   *        </ul>
   * 
   * @return esta instncia.
   */
  public GenericCellEditor setHorizontalAlignment(int alignment) {
    ((JTextField) getComponent()).setHorizontalAlignment(alignment);
    return this;
  }

  /**
   * Apaga qualquer aviso dado ao usurio de que o valor que ele havia inserido
   * era invlido.
   */
  public void clearInvalidValueWarning() {
    ((JComponent) getComponent()).setBorder(new LineBorder(Color.black));
  }

  /**
   * Informa ao usurio que ele est tentando inserir um valor invlida.
   * 
   * @param e Exceo lanada pelo mtodo
   *        {@link #valueOf(String, JTable, int, int)} quando o usurio tenta
   *        submeter um valor invlido.
   * 
   * @see #valueOf(String, JTable, int, int)
   */
  public void warnInvalidValue(Exception e) {
    ((JComponent) getComponent()).setBorder(new LineBorder(Color.red));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean stopCellEditing() {
    String s = (String) super.getCellEditorValue();
    /*
     * Aqui estamos lidando com o caso em que um usurio apagou o valor de uma
     * clula, possivelmente depois que a validao falhou. Retornamos nulo, de
     * modo que o usurio tenha a opo de manter o valor original com a tecla
     * escape, ou substitui-lo por nulo. Para Strings, retornamos "" para manter
     * a compatibilidade com verses anteriores.
     */
    if (s.trim().isEmpty()) {
      if (constructor.getDeclaringClass() == String.class) {
        value = "";
      }
      super.stopCellEditing();
      return true;
    }

    try {
      value = valueOf(s, table, row, column);
    }
    catch (Exception e) {
      warnInvalidValue(e);
      return false;
    }

    if (super.stopCellEditing()) {
      clearInvalidValueWarning();
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void cancelCellEditing() {
    clearInvalidValueWarning();
    super.cancelCellEditing();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellEditorComponent(JTable table, Object value,
    boolean isSelected, int row, int column) {
    this.value = null;
    this.table = table;
    this.column = column;
    this.row = row;

    /**
     * O cdigo abaixo, utilizado para se obter o construtor do tipo da clula,
     * foi colocado aqui, ao invs de estar no mtodo valueOf, por motivo de
     * performance.
     */
    try {
      Class<?> type = table.getColumnClass(column);
      /*
       * Como a nossa obrigao  a de produzir um valor que possa ser
       * convertido para o tipo desejado,  OK usar o construtor de String para
       * colunas que so declaradas objetos, pois um String  um objeto.
       */
      if (type == Object.class) {
        type = String.class;
      }
      this.constructor = type.getConstructor(new Class[] { String.class });
    }
    catch (Exception e) {
      return null;
    }

    return super.getTableCellEditorComponent(table, value, isSelected, row,
      column);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getCellEditorValue() {
    return value;
  }

  /**
   * <p>
   * Extrai de stringValue o valor da clula no tipo esperado pela coluna.
   * </p>
   * <p>
   * Este mtodo tambm pode ser utilizado para validao, lanando uma exceo
   * caso o valor no seja permitido.<br>
   * Por exemplo, se o valor for um percentual, ele deve estar entre 0 e 1.0.<br>
   * <code>
   * protected Object valueOf(String stringValue, JTable table, int row, int column) {
   *   Double value = super.valueOf(stringValue, table, row, column);
   *   if (value > 1.0) {
   *     throw new IllegalArgumentException("value > 1.0");
   *   }
   *   if (value < 0.0) {
   *     throw new IllegalArgumentException("value < 0.0");
   *   }
   * } 
   * </code>
   * </p>
   * 
   * @param stringValue Representao textual do valor da clula.
   * @param table Tabela a qual a clula pertence.
   * @param column ndice da coluna da clula.
   * @param row ndice da coluna da linha.
   * @return o valor da clula extrado de stringValue.
   * 
   * @throws Exception Caso no seja possvel fazer a converso de tipo ou o
   *         valor convertido no seja vlido. Nesse caso o texto da exceo
   *         ser apresentado na statusbar da aplicao e a clula ficar com a
   *         borda vermelha.
   */
  protected Object valueOf(String stringValue, JTable table, int row, int column)
    throws Exception {
    return constructor.newInstance(new Object[] { stringValue });
  }
}
