package csbase.client.util.table;

import java.util.ArrayList;
import java.util.List;

import javax.swing.JComponent;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import csbase.exception.ParseException;

/**
 * Clula de Campo de Texto: a clula que exibe um {@link JTextField} para
 * textos livres.
 */
public class TextFieldCellView extends AbstractCellView {

  /**
   * O conversor.
   */
  private Converter converter;

  /**
   * O campo de texto.
   */
  private JTextField textField;

  /**
   * Validadores.
   */
  private List<Validator> validators;

  /**
   * Cria uma clula de campo de texto.
   */
  public TextFieldCellView() {
    this(HorizontalAlignment.LEFT, null);
  }

  /**
   * Cria uma clula de campo de texto.
   * 
   * @param horizontalAlignment O alinhamento horizontal (Aceita
   *        {@code null} - utiliza-se o 
   *        {@link HorizontalAlignment#getDefault() valor-padro}).
   * @param converter O conversor (Aceita {@code null}).
   * @param validators Os validadores.
   */
  public TextFieldCellView(HorizontalAlignment horizontalAlignment,
    Converter converter, Validator... validators) {
    setValidators(validators);
    setConverter(converter);
    textField = new JTextField();
    textField.setBorder(null);
    textField.getDocument().addDocumentListener(createDocumentListener());
    if (horizontalAlignment == null) {
      horizontalAlignment = HorizontalAlignment.getDefault();
    }
    textField.setHorizontalAlignment(horizontalAlignment.getSwingConstant());
  }

  @Override
  public String getTip() {
    return getValidationErrorMessage();
  }

  public Object getValue() {
    String text = textField.getText();
    if (text.length() == 0) {
      return null;
    }
    try {
      return converter.convertFromView(text);
    } catch (ParseException e) {
      throw new IllegalStateException(e);
    }
  }

  public JComponent getView() {
    return textField;
  }

  @Override
  public void setFocus(boolean hasFocus) {
    super.setFocus(hasFocus);
  }

  @Override
  public void startEditing() {
    super.startEditing();
    textField.getCaret().setVisible(true);
  }

  @Override
  public void stopEditing() {
    textField.getCaret().setVisible(false);
    super.stopEditing();
  }

  @Override
  protected String getValidationErrorMessage() {
    String text = textField.getText();

    Object value;
    if (text.length() != 0) {
      try {
        value = converter.convertFromView(text);
      } catch (ParseException e) {
        return e.getLocalizedMessage();
      }
    }
    else {
      value = null;
    }

    for (Validator validator : validators) {
      String validationErrorMessage = validator
        .getValidationErrorMessage(value);

      if (validationErrorMessage != null) {
        textField.setToolTipText(validationErrorMessage);
        return validationErrorMessage;
      }
    }

    return null;
  }

  @Override
  protected void keepValue(Object value) {
    String textValue = converter.convertToView(value);
    textField.setText(textValue);
  }

  /**
   * Cria um {@link DocumentListener}.
   * 
   * @return .
   */
  private DocumentListener createDocumentListener() {
    return new DocumentListener() {

      public void changedUpdate(DocumentEvent e) {
        fireValueWasChanged();
      }

      public void insertUpdate(DocumentEvent e) {
        fireValueWasChanged();
      }

      public void removeUpdate(DocumentEvent e) {
        fireValueWasChanged();
      }
    };
  }

  /**
   * Atribui um conversor a esta clula.
   * 
   * @param converter O conversor (Aceita {@code null}).
   */
  private void setConverter(Converter converter) {
    if (converter == null) {
      converter = new Converter() {
        public Object convertFromView(String value) {
          return value;
        }

        public String convertToView(Object value) {
          return (String) value;
        }
      };
    }
    this.converter = converter;
  }

  /**
   * Atribui validadores a esta clula.
   * 
   * @param validators Os validadores.
   */
  private void setValidators(Validator... validators) {
    this.validators = new ArrayList<Validator>();
    for (Validator validator : validators) {
      this.validators.add(validator);
    }
  }

  /**
   * O alinhamento.
   */
  public enum HorizontalAlignment {
    RIGHT(SwingConstants.RIGHT),
    LEFT(SwingConstants.LEFT),
    CENTER(SwingConstants.CENTER);

    private int swingConstant;

    HorizontalAlignment(int swingConstant) {
      this.swingConstant = swingConstant;
    }

    /**
     * Obtm o valor-padro ({@link #LEFT}).
     * 
     * @return .
     */
    public static HorizontalAlignment getDefault() {
      return LEFT;
    }
    
    private int getSwingConstant() {
      return swingConstant;
    }
  }
}
