/*
 * $Id:$
 */
package csbase.client.util.xmlpanel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;

import javax.swing.JPanel;
import javax.swing.JSplitPane;

import org.w3c.dom.Node;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.util.xmlpanel.xmlattributetablepanel.XMLAttributeTablePanel;
import csbase.client.util.xmlpanel.xmlsearchpanel.XMLSearchPanel;
import csbase.client.util.xmlpanel.xmltagtreepanel.XMLTagTreePanel;
import csbase.client.util.xmlpanel.xmltextpanel.XMLTextPanel;
import csbase.logic.ClientFile;

/**
 * Painel padro de visualizao XML.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class XMLStandardPanel extends JPanel {

  /**
   * Painel de entidades.
   */
  private final XMLTagTreePanel treePanel = new XMLTagTreePanel();

  /**
   * painel de atributos.
   */
  private final XMLAttributeTablePanel attrPanel = new XMLAttributeTablePanel();

  /**
   * Painel de textos
   */
  private final XMLTextPanel textPanel = new XMLTextPanel();

  /**
   * Painel de busca
   */
  private final XMLSearchPanel searchPanel = new XMLSearchPanel();

  /**
   * Split Pane da aplicao
   */
  private final JSplitPane horizontalSplit = new JSplitPane(
    JSplitPane.HORIZONTAL_SPLIT);

  /**
   * Split Pane da aplicao
   */
  private final JSplitPane verticalSplit = new JSplitPane(
    JSplitPane.VERTICAL_SPLIT);

  /**
   * Arquivo DTD associado.
   */
  private ClientFile dtdClientFile;

  /**
   * Definidor de layout.
   */
  private XMLPanelStyleInterface style;

  /**
   * Ajusta a posio dos split-panes conforme seu default.
   */
  private void adjustDefaultSplitStatus() {
    horizontalSplit.setDividerLocation(0.25);
    if (textPanel.isVisible()) {
      verticalSplit.setTopComponent(attrPanel);
      verticalSplit.setBottomComponent(textPanel);
      horizontalSplit.setRightComponent(verticalSplit);
      verticalSplit.setDividerLocation(0.7);
    }
    else {
      horizontalSplit.setRightComponent(attrPanel);
    }
  }

  /**
   * Consulta  visibilidade do painel de busca.
   * 
   * @return indicativo
   */
  public boolean isSearchPanelVisible() {
    return searchPanel.isVisible();
  }

  /**
   * Atualizao do painel.
   */
  public void refreshAll() {
    treePanel.setNode(treePanel.getNode());
    repaint();
  }

  /**
   * Ajusta n para exibio.
   * 
   * @param dtdClientFile arquivo.
   */
  public void setDTDClientFile(ClientFile dtdClientFile) {
    this.dtdClientFile = dtdClientFile;
  }

  /**
   * Ajuste da visibilidade do painel de busca.
   * 
   * @param flag indicativo
   */
  public void setSearchPanelVisible(boolean flag) {
    searchPanel.setVisible(flag);
  }

  /**
   * Ajusta o definidor do painel de entidades.
   * 
   * @param style definidor
   */
  public void setStyle(XMLPanelStyleInterface style) {
    this.style = style;
    treePanel.setStyle(style);
    updateFromStyle();
    refreshAll();
  }

  /**
   * Atualiza as caractersticas globais do painel com base no estilo.
   */
  private void updateFromStyle() {
    boolean showTextPanel = true;
    if (style != null) {
      showTextPanel = style.isTextPanelNeeded();
    }

    textPanel.setVisible(showTextPanel);
    adjustDefaultSplitStatus();
  }

  /**
   * Consulta o definidor de imagens do painel de entidades.
   * 
   * @return definidor
   */
  public XMLPanelStyleInterface getStyle() {
    return style;
  }

  /**
   * Ajusta n para exibio.
   * 
   * @param xmlClientFile arquivo.
   * @param resolver resolver opcional (se passarmos {@code null}, o mtodo vai
   *        tentar criar um resolver com o arquivo ajustado previamente como
   *        DTD; seno ser usado o resolver definido pelo arquivo XML).
   * @param charset charset.
   * @throws Exception
   */
  public void setXMLClientFile(ClientFile xmlClientFile,
    EntityResolver resolver, Charset charset) throws Exception {
    final EntityResolver er;
    if (resolver != null) {
      er = resolver;
    }
    else {
      if (dtdClientFile != null) {
        er = new EntityResolver() {
          @Override
          public InputSource resolveEntity(String publicId, String systemId)
            throws IOException {
            final InputStream stream = dtdClientFile.getInputStream();
            final InputSource is = new InputSource(stream);
            return is;
          }
        };
      }
      else {
        er = null;
      }
    }

    final Node node = XMLPanelUtils.getRootNode(xmlClientFile, er, charset);
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        treePanel.setNode(node);
        attrPanel.setNode(node);
        textPanel.setNode(node);
        treePanel.setSelectedNode(node);
        horizontalSplit.setDividerLocation(300);
        verticalSplit.setDividerLocation(300);
      }
    });
  }

  /**
   * Construtor
   * 
   * @param attrTitle ttulo da rea de atributos
   * @param textTitle ttulo da rea de texto.
   */
  public XMLStandardPanel(String attrTitle, String textTitle) {
    BorderUtil.setTitledBorder(attrPanel, attrTitle);
    BorderUtil.setTitledBorder(textPanel, textTitle);

    final Dimension minSize = new Dimension(100, 100);
    treePanel.setMinimumSize(minSize);
    attrPanel.setMinimumSize(minSize);
    textPanel.setMinimumSize(minSize);

    final Dimension prefSize = new Dimension(200, 200);
    treePanel.setPreferredSize(prefSize);
    attrPanel.setPreferredSize(prefSize);
    textPanel.setPreferredSize(prefSize);

    setLayout(new BorderLayout());
    verticalSplit.add(attrPanel);
    verticalSplit.add(textPanel);
    verticalSplit.setOneTouchExpandable(true);

    horizontalSplit.add(treePanel);
    horizontalSplit.add(verticalSplit);
    horizontalSplit.setOneTouchExpandable(true);

    adjustDefaultSplitStatus();
    add(horizontalSplit, BorderLayout.CENTER);
    add(searchPanel, BorderLayout.NORTH);

    searchPanel.addSelectionListener(new XMLPanelNodeSelectionListener() {
      @Override
      public void nodeSelected(Node node) {
        treePanel.setSelectedNode(node);
      }
    });

    searchPanel.addComponentListener(new ComponentAdapter() {
      @Override
      public void componentHidden(ComponentEvent e) {
        treePanel.setSelectedNode(treePanel.getNode());
      }
    });

    treePanel.addSelectionListener(new XMLPanelNodeSelectionListener() {
      @Override
      public void nodeSelected(Node node) {
        searchPanel.setNode(node);
        attrPanel.setNode(node);
        textPanel.setNode(node);
      }
    });
  }
}
