package csbase.client;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JFrame;

import csbase.client.ClientSmartFileEvent.OptimizationFailCause;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.logic.ClientOptimizationMode;
import csbase.logic.ClientProjectFile;
import csbase.remote.ClientRemoteLocator;
import tecgraf.javautils.core.lng.LNG;

/**
 * Singleton que tem a funo de instanciar ClientSmartFile a partir de um
 * ClientProjetFile.<br>
 * Considera que o modo e o path no podem ser alterados durante a execuo do
 * cliente. Sendo assim, obtm esses valores uma nica vez, quando 
 * instanciada.<br>
 * Caso o Client receba o parmetro de URL
 * <code>client_optmization_mode = false</code> o modo de otimizao no ser
 * usado.  possvel cadastrar listeners do tipo {@link ClientSmartFileListener}
 * nesta fbrica para gerenciar possveis falhas na utilizao do modo
 * otimizado.
 * 
 *
 * @author Tecgraf
 */
public class ClientSmartFileFactory {

  /**
   * Listeners para eventos relacionados ao modo de otimizao de
   * ClientSmartFile
   */
  List<ClientSmartFileListener> listeners =
    new ArrayList<ClientSmartFileListener>();

  /**
   * Instncia da fbrica
   */
  private static ClientSmartFileFactory instance = null;

  /**
   * Modo de otimizao
   */
  private ClientOptimizationMode optimizationMode = null;

  /**
   * Caminho local para a rea de projetos
   */
  private String localProjectPath = null;

  /**
   * 
   * @return A instncia da fbrica.
   */
  public static ClientSmartFileFactory getInstance() {
    return getInstance(null);
  }

  /**
   * 
   * @param listeners Listeners que sero cadastrados no construtor
   * @return A instncia da fbrica. Cadastra os listeners apenas se for a
   *         primeira chamada ao mtodo getInstance.
   */
  public static ClientSmartFileFactory getInstance(
    List<ClientSmartFileListener> listeners) {
    if (instance == null) {
      instance = new ClientSmartFileFactory(listeners);
    }
    return instance;
  }

  /**
   * 
   * @param cpf Instncia de ClientProjectFile
   * @return Um ClientSmartFile, para acesso otimizado  rea de projetos.
   *         Retorna null, caso cpf seja null.
   */
  public ClientSmartFile create(ClientProjectFile cpf) {
    if (cpf == null) {
      return null;
    }
    ClientSmartFile clientSmartFile =
      new ClientSmartFile(cpf, optimizationMode, localProjectPath);
    return clientSmartFile;
  }

  /**
   * 
   * @return O modo de otimizao para acesso  rea de projetos. Consulta
   *         ProjectService.
   */
  public ClientOptimizationMode getClientOptimizationMode() {
    if (this.optimizationMode != null) {
      return this.optimizationMode;
    }
    RemoteTask<ClientOptimizationMode> task =
      new RemoteTask<ClientOptimizationMode>() {
        @Override
        protected void performTask() throws Exception {
          setResult(ClientRemoteLocator.projectService.getOptimizationMode());
        }
      };
    task.execute(getFrame(), getTitle(), LNG.get("ClientSmartFile.task.msg"));
    if (task.getStatus()) {
      return task.getResult();
    }
    return null;
  }

  /**
   * 
   * @return O path local para acesso otimzados  rea de projetos. Consulta
   *         ProjectService.
   */
  public String getLocalProjectPath() {
    if (localProjectPath != null) {
      return localProjectPath;
    }
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      protected void performTask() throws Exception {
        setResult(ClientRemoteLocator.projectService.getLocalProjectPath());
      }
    };
    task.execute(getFrame(), getTitle(), LNG.get("ClientSmartFile.task.msg"));
    if (task.getStatus()) {
      return task.getResult();
    }
    return null;
  }

  /**
   * 
   * @return o DesktopFrame
   */
  private JFrame getFrame() {
    return DesktopFrame.getInstance().getDesktopFrame();
  }

  /**
   * 
   * @return O Ttulo de DesktopFrame
   */
  private String getTitle() {
    return getFrame().getTitle();
  }

  /**
   * Provisoriamente, esta fbrica obtem dados de otimizao uma vez s no
   * momento da instanciao, assumindo que no sero alterados.
   * 
   * @param listeners Listeners a serem cadastrados no construtor
   */
  public ClientSmartFileFactory(List<ClientSmartFileListener> listeners) {

    /*
     * No verifica nada e no cadastra listeners se o modo no servidor for NONE
     */
    this.optimizationMode = getClientOptimizationMode();
    if (optimizationMode == ClientOptimizationMode.NONE) {
      return;
    }

    if (listeners != null) {
      for (ClientSmartFileListener clientSmartFileListener : listeners) {
        addListener(clientSmartFileListener);
      }
    }

    /* Verifica se Cliente sinaliza que no usa otimizao */
    if (!Client.getInstance().useClientOptimizationMode()) {
      this.optimizationMode = ClientOptimizationMode.NONE;
      this.localProjectPath = null;
      notifyListeners(localProjectPath,
        OptimizationFailCause.CLIENT_PARAMETER_VALUE);
      System.out.println(
        "**** Client executou com parmetro [client_optimization_mode] = false ****");
      return;
    }
    /* Obtm o modo e o path */
    this.localProjectPath = getLocalProjectPath();

    /*
     * Verificando a pasta local de projetos e notificando listeners em caso de
     * no poder usar otimizao
     */
    File localProject = new File(localProjectPath);
    if (!localProject.exists() || !localProject.isDirectory()) {
      notifyListeners(localProjectPath,
        OptimizationFailCause.PROJECT_PATH_NOT_VALID);
      this.optimizationMode = ClientOptimizationMode.NONE;
      System.out.println(
        "**** O path para a rea de projetos no  vlido. Desabilitando otimizao. ****");
    }
    else if (!localProject.canRead()) {
      notifyListeners(localProjectPath,
        OptimizationFailCause.CANT_READ_PROJECT_PATH);
      this.optimizationMode = ClientOptimizationMode.NONE;
      System.out.println(
        "**** O path para a rea de projetos no permite leitura. Desabilitando otimizao. ****");
    }
    else {
      System.out
        .println("ClientSmartFileFactory instanciada:\nModo de Otimizao: "
          + optimizationMode.name() + "\nrea Local de Projeto: "
          + localProjectPath);
    }
  }

  /**
   * 
   * @return true, se est em modo de otiizao
   */
  public boolean isOptimizedMode() {
    return optimizationMode == ClientOptimizationMode.GLOBAL;
  }

  /**
   * Adiciona um listener
   * 
   * @param listener
   */
  public void addListener(ClientSmartFileListener listener) {
    listeners.add(listener);
  }

  /**
   * Remove um listener
   * 
   * @param listener
   */
  public void removeListener(ClientSmartFileListener listener) {
    listeners.remove(listener);
  }

  /**
   * Notifica listeners sobre falha na otimizao de arquivo
   * 
   * @param localProjectPath Path local da rea de projetos
   * @param localFilePath Path local do arquivo smart
   * @param cause Causa da notificao
   */
  void notifyListeners(String localProjectPath, String localFilePath,
    OptimizationFailCause cause) {
    for (ClientSmartFileListener clientSmartFileListener : listeners) {
      clientSmartFileListener.clientOptimizationFileFailed(
        new ClientSmartFileEvent(localProjectPath, localFilePath, cause));
    }
  }

  /**
   * Notifica listeners sobre falha na otimizao
   * 
   * @param localProjectPath Path local da rea de projetos
   * @param cause Causa da notificao
   */
  void notifyListeners(String localProjectPath, OptimizationFailCause cause) {
    for (ClientSmartFileListener clientSmartFileListener : listeners) {
      clientSmartFileListener.clientOptimizationFailed(
        new ClientSmartFileEvent(localProjectPath, cause));
    }
  }
}
