package csbase.client.applications.algorithmsmanager.actions;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import csbase.client.applications.algorithmsmanager.dialogs.CommonSelectionPanel;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.AlgorithmInfo;

/**
 * Essa classe representa a ao de remover um ou mais algoritmos, na
 * funcionalidade de gerenciamento de algoritmos.
 * 
 * O mtodo <code> handleEditOperation </code> deve ser redefinido para que seja
 * confirmado pelo usurio a remoo de todos os algoritmos selecionados no
 * painel de seleo.
 * 
 */
public class AlgorithmRemoveAction extends CommonEditAction {
  /**
   * Constri a ao de remover algoritmos.
   * 
   * @param selectionPanel painel pai que criou a ao
   * @param icon imagem da ao
   */
  public AlgorithmRemoveAction(CommonSelectionPanel selectionPanel,
    ImageIcon icon) {
    super(selectionPanel, icon);
  }

  /**
   * Remove efetivamente os algoritmos no servidor.
   * 
   * @param algorithms lista com os algoritmos a serem removidos
   * @return o nmero de algoritmos que foram efetivamente removidos
   * 
   */
  private int removeAlgorithms(final List<AlgorithmInfo> algorithms) {
    int removedCounter = 0;
    for (AlgorithmInfo algoInfo : algorithms) {
      boolean removed =
        AlgorithmManagementProxy.removeAlgorithm(algoInfo.getId(),
          getApplication().getApplicationFrame());
      if (removed) {
        removedCounter++;
      }
    }
    return removedCounter;
  }

  @Override
  protected void handleEditOperation() {
    List<DataInterface> algorithmItems =
      getSelectionPanel().getSelectedDataList();
    if (algorithmItems == null || algorithmItems.isEmpty()) {
      return;
    }

    int option =
      getApplication().showOptionDialog(
        getString("AlgorithmRemoveAction.msg.cancel.confirm"),
        new String[] {
            getString("AlgorithmRemoveAction.msg.cancel.confirm.yes"),
            getString("AlgorithmRemoveAction.msg.cancel.confirm.no") });
    if (option == 0) {
      int removedCounter = removeAlgorithms(getAlgorithms(algorithmItems));
      if (removedCounter != algorithmItems.size()) {
        String msg =
          MessageFormat.format(
            getString("AlgorithmRemoveAction.msg.error.remove.no_sucess"),
            new Object[] { removedCounter });
        getApplication().showMessage(msg);
      }
    }
  }

  /**
   * Obtm uma lista de algoritmos a partir de uma lista de itens de algoritmos
   * selecionados para remoo.
   * 
   * @param algorithmItemList lista de itens de algoritmos
   * @return a lista de algoritmos correspondente aos itens selecionados
   */
  private List<AlgorithmInfo> getAlgorithms(
    List<DataInterface> algorithmItemList) {
    List<AlgorithmInfo> algoList = new ArrayList<AlgorithmInfo>();
    for (DataInterface item : algorithmItemList) {
      AlgorithmListItem algoItem = (AlgorithmListItem) item;
      algoList.add(algoItem.getItem());
    }
    return algoList;
  }

}
