package csbase.client.applications.projectsmanager.proxy;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.MailServiceInterface;

/**
 * Task responsvel pelo envio de mails a usurios, referentes a projetos.
 * 
 * @author jnlopes
 */
public class AdminSendEmailTask extends ProjectsManagerTask<String[]> {

  /**
   * Nomes dos projetos.
   */
  final private Map<Object, Set<String>> userProjectMap;

  /**
   * Assunto do email a ser enviado.
   */
  final private String mailSubject;

  /**
   * Contedo do email a ser enviado.
   */
  final private String mailText;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   * @param userProjectMap HashMap contendo a relao (1-N) Usurio - Projetos.
   * @param mailSubject Assunto do email a ser enviado.
   * @param mailText Mail a ser enviado aos usurios.
   */
  public AdminSendEmailTask(final ProjectsManager projectsManager,
    final Map<Object, Set<String>> userProjectMap, final String mailSubject,
    final String mailText) {
    super(projectsManager);
    this.userProjectMap = userProjectMap;
    this.mailSubject = mailSubject;
    this.mailText = mailText;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() {
    final List<Object> list = new ArrayList<Object>();
    final MailServiceInterface msi = ClientRemoteLocator.mailService;
    final Set<Object> users = userProjectMap.keySet();
    final User u = User.getLoggedUser();
    final String[] ownMails = u.getEmails();
    if (ownMails == null || ownMails.length == 0) {
      setResult(list.toArray(new String[0]));
      return;
    }

    for (final Object user : users) {
      final String ajustedMailText = adjustMailText(user);
      try {
        final User us = User.getUser(user);
        if (us == null) {
          list.add(user);
          continue;
        }
        final String[] userMails = us.getEmails();
        if (userMails == null || userMails.length == 0) {
          list.add(user);
          continue;
        }
        msi.sendMail(ownMails[0], userMails, ajustedMailText, mailSubject,
          null, null);
      }
      catch (final RemoteException re) {
        list.add(user);
      }
    }

    if (list.size() > 0) {
      setResult(list.toArray(new String[list.size()]));
    }
    else {
      setResult(null);
    }
  }

  /**
   * Inclui mensagem final ao email, informando os projetos relativos a
   * mensagem.
   * 
   * @param userId Id do usurio para quem o mail ser enviado.
   * @return Mail modificado.
   */
  private String adjustMailText(final Object userId) {
    if (userId == null) {
      return "";
    }

    final StringBuilder defaultNote =
      new StringBuilder(getString("AdminSendEmailTask.default.mail.note.text"));

    final Set<String> projects = userProjectMap.get(userId);
    if (projects == null) {
      return "";
    }

    for (final String str : projects) {
      defaultNote.append(" " + str + ",");
    }
    return mailText + "\n" + defaultNote.substring(0, defaultNote.length() - 1);
  }
}
