/**
 * $Id: RemoteTask.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.desktop;

import java.awt.Dialog.ModalityType;
import java.rmi.RemoteException;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.ClientServerManager;
import csbase.client.algorithms.validation.ValidationTranslator;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.BugException;
import csbase.exception.CSBaseRuntimeException;
import csbase.exception.InfoException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.exception.UnavailableServiceException;
import csbase.exception.algorithms.AlgorithmValidationException;
import csbase.logic.ServerURI;
import csbase.logic.algorithms.validation.Validation;

/**
 * A classe <code>RemoteTask</code> modela uma tarefa executada no cliente que
 * envolve chamadas a mtodos remotos. As excees que representam falhas na
 * execuo de chamadas remotas so tratadas, exibindo-se mensagens de erro para
 * o usurio. Tambm  tratada a interrupo da tarefa, resultante de um
 * cancelamento comandado pelo usurio.
 * 
 * @param <R> A classe do resultado da tarefa.
 */
public abstract class RemoteTask<R> extends Task<R> {

  /** conexo com o servidor */
  protected ServerURI serverURI;

  /**
   * Constri uma task com bloqueio de janelas
   * {@link ModalityType#APPLICATION_MODAL}
   */
  protected RemoteTask() {
    this(ModalityType.APPLICATION_MODAL);
  }

  /**
   * Construtor.
   * 
   * @param modality tipo de bloqueio das janelas
   */
  public RemoteTask(ModalityType modality) {
    this(null, modality);
  }

  /**
   * Construtor.
   * 
   * @param serverURI conexo com o servidor.
   */
  protected RemoteTask(ServerURI serverURI) {
    this(serverURI, ModalityType.APPLICATION_MODAL);
  }

  /**
   * Construtor.
   * 
   * @param serverURI conexo com o servidor
   * @param modality tipo de bloqueio das janelas
   */
  protected RemoteTask(ServerURI serverURI, ModalityType modality) {
    super(modality);
    this.serverURI = serverURI;
  }

  /**
   * Tratamento de erros especfico para cada tipo de tarefa. Esse tratamento 
   * chamado na thread do Swing. Pode ser redefinido por uma especializao
   * dessa classe, porm com cautela, para que a padronizao do tratamento de
   * erros no seja quebrada!
   * 
   * @param error exceo que representa erro durante execuo da tarefa
   */
  @Override
  protected void handleError(Exception error) {
    /* erros do servidor */
    if (error instanceof InfoException) {
      handleInfoError((InfoException) error);
    }
    else if (error instanceof CSBaseRuntimeException) {
      handleServerError((CSBaseRuntimeException) error);
    }
    /* falha no RMI */
    else if (error instanceof RemoteException) {
      StandardErrorDialogs.showErrorDialog(parentWindow, taskTitle,
        LNG.get("desktop.msg.commfailure"), error, getAdditionalInfo());
      if (this.serverURI != null) {
        ClientServerManager.getInstance().invalidate(this.serverURI);
      }
      else {
        ClientServerManager.getInstance().invalidate();
      }
    }
    else {
      StandardErrorDialogs.showErrorDialog(parentWindow, taskTitle,
        LNG.get("desktop.msg.clientbug"), error, getAdditionalInfo());
    }
  }

  /**
   * Tratamento para erros repassados pelo proxy do servio remoto.
   * 
   * @param cserror exceo que representa esse erro
   */
  protected void handleServerError(CSBaseRuntimeException cserror) {
    String msg = cserror.getMessage();
    if (msg == null) {
      if (cserror instanceof UnavailableServiceException) {
        msg = LNG.get("desktop.msg.unavailable");
      }
      else if (cserror instanceof ServiceFailureException) {
        msg = LNG.get("desktop.msg.servicefailure");
      }
      else if (cserror instanceof PermissionException) {
        msg = LNG.get("desktop.msg.permissionfailure");
      }
      else if (cserror instanceof AlgorithmValidationException) {
        msg = LNG.get("desktop.msg.validationfailure");
        Validation validation =
          ((AlgorithmValidationException) cserror).getValidationResult();
        msg += ValidationTranslator.translateMessage(validation);
      }
      else if (cserror instanceof BugException) {
        msg = LNG.get("desktop.msg.serverbug");
      }
      else {
        msg = LNG.get("desktop.msg.unknownerror");
      }
    }
    StandardErrorDialogs.showErrorDialog(parentWindow, taskTitle, msg, cserror,
      getAdditionalInfo());
  }

  /**
   * Tratamento de erros que representam situaes previsveis e no so erros
   * de fato.
   * 
   * @param infoError exceo que representa esse erro
   */
  protected void handleInfoError(InfoException infoError) {
    StandardErrorDialogs.showErrorDialog(parentWindow, taskTitle,
      infoError.getMessage());
  }
}
