/*
 * Detalhes da ltima alterao:
 * 
 * $Author: clinio $ $Date: 2010-08-10 17:18:27 -0300 (Tue, 10 Aug 2010) $
 * $Revision: 108897 $
 */
package csbase.client.login;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;

import javax.swing.ComboBoxModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.configurationmanager.MissingPropertyException;
import csbase.exception.ConfigurationException;

/**
 * Representa um modelo para combo boxes de locale.
 * 
 * @author Tecgraf/PUC-Rio
 */
final class LocaleComboBoxModel implements ComboBoxModel {
  /**
   * <p>
   * Um ndice invlido.  usado para gerar o evento de alterao no contedo do
   * modelo ou na seleo.
   * </p>
   * 
   * <p>
   * Quando muda-se a seleo,  necessrio gerar um evento de mudana no
   * contedo do modelo e o ndice do elemento alterado no deve ser um ndice
   * de elemento vlido.
   * </p>
   */
  private static final int INVALID_INDEX = -1;

  /**
   * Representa o nome da propriedade que contm o nome do locale, a ser exibido
   * para o usurio.
   */
  private static final String DISPLAY_PROPERTY = "locale.display";

  /**
   * Representa o nome da propriedade que contm o nome do locale, a ser
   * utilizado programaticamente.
   */
  private static final String PROGRAMMATIC_PROPERTY = "locale.programmatic";

  /**
   * Representa o nome da propriedade que contm o nome do locale, a ser exibido
   * para o usurio. S  definido nas instncias do framework.
   */
  private static final String ADDITIONAL_DISPLAY_PROPERTY = "additional.locale.display";

  /**
   * Representa o nome da propriedade que contm o nome do locale, a ser exibido
   * para o usurio. S  definido nas instncias do framework.
   */
  private static final String ADDITIONAL_PROGRAMMATIC_PROPERTY = "additional.locale.programmatic";

  /**
   * Uma lista com os locales disponveis no modelo.
   */
  private List<LocaleComboBoxItem> localeList;

  /**
   * O item atualmente selecionado.
   */
  private LocaleComboBoxItem selectedLocale;

  /**
   * Uma lista com os ouvintes interessados em alteraes no modelo.
   */
  private List<ListDataListener> listDataListenerList;

  /**
   * Cria um modelo para combo boxes de locale.
   * 
   * @throws ConfigurationException Caso no seja possvel obter a configurao
   *         da classe.
   * @throws IllegalStateException Caso no tenha sido definido nenhum idioma
   *         nas configuraes da classe.
   */
  LocaleComboBoxModel() {
    this.listDataListenerList = new LinkedList<ListDataListener>();
    Configuration configuration;
    try {
      configuration = ConfigurationManager.getInstance().getConfiguration(
        LocaleComboBoxModel.class);
    }
    catch (ConfigurationManagerException e) {
      throw new ConfigurationException(e);
    }
    List<String> nameList;
    List<Locale> programmaticList;
    try {
      nameList = configuration.getMandatoryListProperty(DISPLAY_PROPERTY);
      programmaticList = configuration
        .getMandatoryLocaleListProperty(PROGRAMMATIC_PROPERTY);
    }
    catch (MissingPropertyException e) {
      e.printStackTrace();
      throw new IllegalStateException("No foi definido nenhum idioma.");
    }
    List<String> additionalNameList = configuration
      .getOptionalListProperty(ADDITIONAL_DISPLAY_PROPERTY);
    List<Locale> additionalLanguageList = configuration
      .getOptionalLocaleListProperty(ADDITIONAL_PROGRAMMATIC_PROPERTY);
    this.localeList = new ArrayList<LocaleComboBoxItem>(nameList.size()
      + additionalNameList.size());
    for (int i = 0; i < nameList.size(); i++) {
      this.localeList.add(new LocaleComboBoxItem(nameList.get(i),
        programmaticList.get(i)));
    }
    for (int i = 0; i < additionalNameList.size(); i++) {
      this.localeList.add(new LocaleComboBoxItem(additionalNameList.get(i),
        additionalLanguageList.get(i)));
    }
    this.selectedLocale = this.localeList.get(0);
  }

  /**
   * Obtm o item selecionado que  um objeto da classe {@link
   * LocaleComboBoxItem}.
   * 
   * @see ComboBoxModel#getSelectedItem()
   */
  @Override
  public Object getSelectedItem() {
    return this.selectedLocale;
  }

  /**
   * Define o objeto atualmente selecionado. Este objeto tem que ser da classe
   * {@link LocaleComboBoxItem}.
   * 
   * @see ComboBoxModel#setSelectedItem(java.lang.Object)
   */
  @Override
  public void setSelectedItem(Object anItem) {
    int index = this.localeList.indexOf(anItem);
    if (index == -1) {
      throw new IllegalArgumentException("O item no existe no modelo.");
    }
    this.selectedLocale = this.localeList.get(index);
    this.fireContentsChanged(INVALID_INDEX, INVALID_INDEX);
  }

  /**
   * Informa  todos os ouvintes cadastrados que houve uma mudana no contedo
   * do modelo, informando o intervalo de ndices dos elementos que fazem parte
   * das mudanas.
   * 
   * @param firstIndex O ndice do primeiro elemento alterado.
   * @param lastIndex O ndice do ltimo elemento alterado.
   */
  private void fireContentsChanged(int firstIndex, int lastIndex) {
    ListDataEvent listDataEvent = new ListDataEvent(this,
      ListDataEvent.CONTENTS_CHANGED, firstIndex, lastIndex);
    Iterator<ListDataListener> listDataListenerIterator = this.listDataListenerList.iterator();
    while (listDataListenerIterator.hasNext()) {
      ListDataListener listDataListener = listDataListenerIterator.next();
      listDataListener.contentsChanged(listDataEvent);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getSize() {
    return this.localeList.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getElementAt(int index) {
    return this.localeList.get(index);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addListDataListener(ListDataListener listDataListener) {
    if (listDataListener == null) {
      throw new IllegalArgumentException("O ouvinte no pode ser nulo.");
    }
    this.listDataListenerList.add(listDataListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeListDataListener(ListDataListener listDataListener) {
    if (listDataListener == null) {
      throw new IllegalArgumentException("O ouvinte no pode ser nulo.");
    }
    this.listDataListenerList.remove(listDataListener);
  }
}
