/*
 * $Id: TablePanel.java,v 1.1 2010/06/12 00:10:47 clinio Exp $
 */

package csbase.client.util.filechooser.filetablepanel;

import java.awt.BorderLayout;
import java.awt.Rectangle;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.core.string.StringUtils;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.util.filechooser.ClientFileChooserCardinality;
import csbase.client.util.filechooser.ClientFileChooserSelectionMode;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.logic.ClientFile;

/**
 * Painel de exibio de diretrios
 *
 * @author Tecgraf/PUC-Rio
 */
public class ClientFileTablePanel extends JPanel {

  /**
   * rvore
   */
  private final SortableTable table = new SortableTable();

  /**
   * Indicativo de mode de seleo.
   */
  private ClientFileChooserSelectionMode selectionMode =
    ClientFileChooserSelectionMode.FILES_ONLY;

  /**
   * Modelo.
   */
  private final ClientFileTableModel model;

  /**
   * Listeners de mouse
   */
  final private List<ClientTableActionListener> actionListeners =
    new ArrayList<ClientTableActionListener>();

  /**
   * Listeners de mouse
   */
  final private List<ClientFileTableSelectionListener> selectionListeners =
    new ArrayList<ClientFileTableSelectionListener>();

  /**
   * Flag de seleo simples
   */
  private ClientFileChooserCardinality cardinality =
    ClientFileChooserCardinality.SINGLE_CHOOSE;

  /**
   * Creates a new <code>FileExplorer</code> instance.
   *
   */
  public ClientFileTablePanel() {
    super();
    setLayout(new BorderLayout());
    model = new ClientFileTableModel(null, null, table);
    table.setModel(model);
    table.setGridColor(table.getBackground());
    table.setFillsViewportHeight(true);

    initColumns();
    initComparators();

    table.setColumnSelectionAllowed(false);
    table.setRowSelectionAllowed(true);
    setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);

    final ClientFileTableRenderer renderer = new ClientFileTableRenderer();
    table.setDefaultRenderer(ClientFile.class, renderer);

    addTableKeyListener();
    addTableActionListener();
    addTableSelectionListener();

    final JScrollPane scroll = new JScrollPane(table);
    add(scroll, BorderLayout.CENTER);
  }

  /**
   * Adiciona listener de teclado para facilitar a busca de arquivos por sua
   * letra inicial.
   */
  private void addTableKeyListener() {
    table.addKeyListener(new KeyAdapter() {

      /**
       * O caractere atual.
       */
      private char currentChar = ' ';

      /**
       * Os arquivos que comeam com o caractere atual.
       */
      private List<ClientFile> files = Collections.emptyList();

      /**
       * O ndice atual para percorrer a lista de arquivos.
       */
      private int index = 0;

      /**
       * O diretrio atual.
       */
      private ClientFile currentDir = null;

      @Override
      public void keyTyped(KeyEvent e) {
        char c = e.getKeyChar();
        ClientFile dir = model.getDirectory();
        /*
         * Se o caractere e o diretrio escolhidos so os mesmos, atualiza s o
         * ndice da lista de arquivos.
         */
        if (c == currentChar && dir.equals(currentDir)) {
          index++;
        }
        // Seno busca a nova lista de arquivos.
        else {
          index = 0;
          currentChar = c;
          currentDir = dir;
          // S aceita os caracteres padro para nome de arquivo.
          if (Character.isLetterOrDigit(c) || c == '_' || c == '.' || c == '-') {
            files = model.getFilesWithPrefix(String.valueOf(c));
          }
          else {
            files = Collections.emptyList();
          }
        }
        int numFiles = files.size();
        if (numFiles > 0) {
          // ndice volta pro incio depois de percorrer toda a lista
          int i = index % numFiles;
          ArrayList<ClientFile> selection = new ArrayList<ClientFile>();
          selection.add(files.get(i));
          setSelection(selection);
          scrollToSelection();
        }
      }
    });
  }

  /**
   * Garante que o item selecionado esteja visvel.
   */
  protected void scrollToSelection() {
    int selectedRow = table.getSelectedRow();
    if (selectedRow >= 0) {
      Rectangle cellRect = table.getCellRect(selectedRow, 0, true);
      table.scrollRectToVisible(cellRect);
    }
  }

  /**
   * Inicializa as colunas.
   */
  private void initColumns() {
    final TableColumnModel columnModel = table.getColumnModel();
    columnModel.getColumn(0).setPreferredWidth(40);
    columnModel.getColumn(1).setPreferredWidth(220);
    columnModel.getColumn(2).setPreferredWidth(110);
    columnModel.getColumn(3).setPreferredWidth(150);
    table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
    final JTableHeader header = table.getTableHeader();
    header.setReorderingAllowed(false);
  }

  /**
   * Adiciona listeners da rvore.
   */
  private void addTableActionListener() {
    final ClientFileTablePanel panel = this;
    table.addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(final MouseEvent me) {
        if (me.isControlDown() || me.isShiftDown()) {
          return;
        }
        final boolean isLeft = SwingUtilities.isLeftMouseButton(me);
        final int clickCount = me.getClickCount();
        final List<ClientFile> objects = getSelectedClientProjectFiles();
        if (objects == null) {
          return;
        }

        if (isLeft && clickCount == 2) {
          for (final ClientTableActionListener lst : actionListeners) {
            lst.actionPerformed(panel, objects);
          }
        }
      }
    });
  }

  /**
   * Adiciona listener de mouse.
   *
   * @param listener o listener
   */
  final public void addTablePanelActionListener(
    final ClientTableActionListener listener) {
    actionListeners.add(listener);
  }

  /**
   * Adiciona listener de mouse.
   *
   * @param listener o listener
   */
  final public void addTablePanelSelectionListener(
    final ClientFileTableSelectionListener listener) {
    selectionListeners.add(listener);
  }

  /**
   * Adiciona listeners da rvore.
   */
  private void addTableSelectionListener() {
    final ClientFileTablePanel panel = this;
    table.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        @Override
        public void valueChanged(final ListSelectionEvent e) {
          final List<ClientFile> objects = getSelectedClientProjectFiles();
          if (objects == null || objects.size() == 0) {
            return;
          }
          for (final ClientFileTableSelectionListener lst : selectionListeners) {
            lst.selectionPerformed(panel, objects);
          }
        }
      });
  }

  /**
   * Remove listener de mouse.
   *
   * @param listener o listener
   */
  final public void delTablePanelActionListener(
    final ClientTableActionListener listener) {
    actionListeners.remove(listener);
  }

  /**
   * Adiciona listener de seleo.
   *
   * @param listener o listener
   */
  final public void delTablePanelSelectionListener(
    final ClientFileTableSelectionListener listener) {
    selectionListeners.remove(listener);
  }

  /**
   * Consulta a lista de objetos na tabela.
   *
   * @return a lsita
   */
  final public List<ClientFile> getClientProjectFileList() {
    final List<ClientFile> objects = new ArrayList<ClientFile>();
    final List<ClientFile> objectList = model.getFileList();
    objects.addAll(objectList);
    Collections.unmodifiableList(objects);
    return objects;
  }

  /**
   * Consulta o valor de: diretrio raiz.
   *
   * @return o valor.
   */
  public final ClientFile getRootDirectory() {
    return model.getDirectory();
  }

  /**
   * Consulta a lista de objetos selecionados.
   *
   * @return a lista
   */
  final public List<ClientFile> getSelectedClientProjectFiles() {
    final List<ClientFile> list = new ArrayList<ClientFile>();
    final List<ClientFile> fileList = model.getFileList();
    final int[] selectedRows = table.getSelectedRows();
    for (final int idx : selectedRows) {
      final int realIdx = table.convertRowIndexToModel(idx);
      final ClientFile cpf = fileList.get(realIdx);
      list.add(cpf);
    }
    return list;
  }

  /**
   * Consulta ao modo de seleo para arquivos.
   *
   * @return modo
   */
  final public ClientFileChooserSelectionMode getSelectionMode() {
    return this.selectionMode;
  }

  /**
   * Monta comparadores.
   */
  private void initComparators() {
    table.setNoSortStateEnabled(false);
    table.sort();

    final Comparator<ClientFile> typeComparator = new Comparator<ClientFile>() {
      @Override
      public int compare(final ClientFile o1, final ClientFile o2) {
        final boolean equalStructs = (o1.isDirectory() == o2.isDirectory());
        if (!equalStructs) {
          return o1.isDirectory() ? -1 : 1;
        }
        final String t1 = o1.getType();
        final String t2 = o2.getType();
        final boolean equalTypes = t1.equals(t2);
        if (!equalTypes) {
          return t1.compareTo(t2);
        }
        final SortOrder sortOrder = table.getCurrentSortOrder();
        final boolean isAscending = (sortOrder == SortOrder.ASCENDING);

        final String n1 = o1.getName();
        final String n2 = o2.getName();
        final Comparator<String> plain = StringUtils.getPlainSortComparator();
        final int cmp = plain.compare(n1, n2);
        return isAscending ? cmp : -cmp;
      }
    };

    final Comparator<ClientFile> sizeComparator = new Comparator<ClientFile>() {
      @Override
      public int compare(final ClientFile o1, final ClientFile o2) {
        final boolean equalStructs = (o1.isDirectory() == o2.isDirectory());
        if (equalStructs) {
          final Long c1 = o1.size();
          final Long c2 = o2.size();
          return c1.compareTo(c2);
        }
        return o1.isDirectory() ? -1 : 1;
      }
    };

    final Comparator<ClientFile> modComparator = new Comparator<ClientFile>() {
      @Override
      public int compare(final ClientFile o1, final ClientFile o2) {
        final Long c1 = o1.getModificationDate();
        final Long c2 = o2.getModificationDate();
        return c1.compareTo(c2);
      }
    };

    final Comparator<ClientFile> nameComparator = new Comparator<ClientFile>() {
      @Override
      public int compare(final ClientFile o1, final ClientFile o2) {
        final SortOrder sortOrder = table.getCurrentSortOrder();
        final boolean isAscending = (sortOrder == SortOrder.ASCENDING);

        final boolean isDir1 = o1.isDirectory();
        final boolean isDir2 = o2.isDirectory();
        final String n1 = o1.getName();
        final String n2 = o2.getName();
        if (isDir1 && !isDir2) {
          return (isAscending ? -1 : 1);
        }
        if (!isDir1 && isDir2) {
          return (isAscending ? 1 : 1);
        }
        final Comparator<String> plain = StringUtils.getPlainSortComparator();
        return plain.compare(n1, n2);
      }
    };

    table.setComparator(0, typeComparator);
    table.setComparator(1, nameComparator);
    table.setComparator(2, sizeComparator);
    table.setComparator(3, modComparator);
  }

  /**
   * Consulta se modo de seleo  simples.
   *
   * @return indicativo
   */
  final public ClientFileChooserCardinality getCardinality() {
    return cardinality;
  }

  /**
   * Ajuste de visualizao.
   *
   * @param filter filtro
   */
  final public void setFilter(final ClientFileFilterInterface filter) {
    model.setFilter(filter);
  }

  /**
   * Ajuste de diretrio.
   *
   * @param directory diretrio
   */
  final public void setRootDirectory(final ClientFile directory) {
    model.setDirectory(directory);
  }

  /**
   * Ajuste de seleo.
   *
   * @param newSelection seleo
   */
  public void setSelection(final List<ClientFile> newSelection) {
    final int nRows = model.getRowCount();
    table.clearSelection();
    if (newSelection == null) {
      return;
    }

    ListSelectionModel selectionModel = table.getSelectionModel();
    for (int r = 0; r < nRows; r++) {
      final Object obj = table.getValueAt(r, 0);
      final ClientFile file = (ClientFile) obj;
      if (newSelection.contains(file)) {
        selectionModel.addSelectionInterval(r, r);
      }
    }
  }

  /**
   * Ajuste do modo de seleo.
   *
   * @param mode modo
   */
  final public void setSelectionMode(final ClientFileChooserSelectionMode mode) {
    this.selectionMode = mode;
    table.clearSelection();
  }

  /**
   * Ajuste do modo de seleo simples.
   *
   * @param flag indicativo
   */
  final public void setCardinality(final ClientFileChooserCardinality flag) {
    final int tableSelection;
    switch (flag) {
      case SINGLE_CHOOSE:
        tableSelection = ListSelectionModel.SINGLE_SELECTION;
        break;
      case MULTIPLE_CHOOSE:
        tableSelection = ListSelectionModel.MULTIPLE_INTERVAL_SELECTION;
        break;
      default:
        throw new IllegalArgumentException("bad value!");
    }
    cardinality = flag;
    table.setSelectionMode(tableSelection);
    table.clearSelection();
  }

  /**
   * Atualizao da tabela.
   */
  final public void refresh() {
    model.refresh();
  }

  /**
   * Indica que arquivos ocultos devem ser exibidos.
   *
   * @param flag indicativo
   */
  final public void showHiddenFiles(boolean flag) {
    model.showHiddenFiles(flag);
    refresh();
  }

  /**
   * Indica se arquivos ocultos devem ser vistos.
   *
   * @return indicativo
   */
  public final boolean isHiddenFilesShown() {
    return model.isHiddenFilesShown();
  }

}
