package csbase.client.util.gui.log.tab;

import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeSet;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.ScrollPaneConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.Client;
import csbase.client.ClientLocalFile;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.util.event.EventListener;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.client.util.gui.log.LogPanelReloader;
import csbase.client.util.gui.log.LogPanelReloader.StatusChangedEvent;
import csbase.client.util.gui.log.LogPanelTextArea;
import csbase.client.util.iostring.TextReader;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;

/**
 * Aba para visualizao de logs consolidados (vrios arquivos de log
 * simultaneamente).
 */
public class ConsolidatedLogTab extends AbstractLogTab {

  /**
   * reas de texto do log, mapeadas pelo arquivo de log sendo visualizado.
   */
  private final Map<ClientProjectFile, LogPanelTextArea> panels;

  /**
   * Painel principal da aba.
   */
  private JPanel mainPanel;

  /**
   * Guarda o ltimo diretrio usado na exportao.
   */
  private ClientLocalFile lastSelectedDir;

  /**
   * Painel rolvel que contem os painis de log.
   */
  private JScrollPane scrollPane;

  /**
   * Construtor.
   *
   * @param title ttulo da aba.
   * @param files conjunto de arquivos de log.
   * @param statusBar barra de status para apresentar mensagens da aba.
   * @param owner janela dona da aba.
   * @param autoReload indica se a aba deve ser recarregada automaticamente.
   */
  public ConsolidatedLogTab(final String title,
    final Set<ClientProjectFile> files, final StatusBar statusBar,
    Window owner, boolean autoReload) {
    super(title, statusBar, owner, autoReload);
    panels = new HashMap<ClientProjectFile, LogPanelTextArea>();
    mainPanel = new JPanel();
    mainPanel.setLayout(new GridBagLayout());

    initUI(files);
  }

  /**
   * Inicializa os componentes da aba.
   *
   * @param files conjunto de arquivos de log.
   */
  private void initUI(final Set<ClientProjectFile> files) {
    JPanel internalPanel = new JPanel();
    internalPanel.setLayout(new GridBagLayout());
    int i = 0;
    for (final ClientProjectFile file : files) {
      final LogPanelTextArea textArea =
        new LogPanelTextArea(this, false, "[" + file.getName() + "]");
      textArea.setPrefix(0);
      JSeparator separator = new JSeparator();
      panels.put(file, textArea);
      internalPanel.add(separator, new GBC(0, i++).horizontal());
      internalPanel.add(textArea, new GBC(0, i++).both());
    }
    JToolBar toolbar = createToolBar();
    mainPanel.add(toolbar, new GBC(0, 0).horizontal());
    scrollPane = new JScrollPane(internalPanel);
    final int mode = ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS;
    scrollPane.setVerticalScrollBarPolicy(mode);
    mainPanel.add(scrollPane, new GBC(0, 1).both());
    reload();
  }

  /**
   * Cria a barra de ferramentas.
   *
   * @return a barra de ferramentas.
   */
  private JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable(false);
    final JToggleButton autoReloadButton = new JToggleButton();
    autoReloadButton.setAction(new ReloadAction());
    autoReloadButton.setSelected(reloader.isRunning());
    autoReloadButton.setToolTipText(LNG.get(ConsolidatedLogTab.class
      .getSimpleName()
      + ".button.reload.tooltip"));
    reloader
    .addStatusChangedEventListener(new EventListener<LogPanelReloader.StatusChangedEvent>() {
      @Override
      public void eventFired(StatusChangedEvent event) {
        autoReloadButton.setSelected(event.isRunning());
      }
    });
    toolBar.add(autoReloadButton);
    toolBar.addSeparator();

    final JButton exportButton = new JButton();
    exportButton.setAction(new ExportAction());
    exportButton.setToolTipText(LNG.get(ConsolidatedLogTab.class
      .getSimpleName()
      + ".button.export.tooltip"));
    toolBar.add(exportButton);

    return toolBar;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void reload() {
    RemoteTask<Map<ClientProjectFile, String>> reloadTask =
      new RemoteTask<Map<ClientProjectFile, String>>() {
        @Override
        protected void performTask() throws Exception {
          Map<ClientProjectFile, String> result =
            new HashMap<ClientProjectFile, String>();
          for (ClientProjectFile file : panels.keySet()) {
            String fileContent = readFile(file);
            result.put(file, fileContent);
          }
          setResult(result);
        }

      @Override
      protected void afterTaskUI() {
        Exception error = getError();
        if (error != null) {
          String errorMsg =
            LNG.get(ConsolidatedLogTab.class.getSimpleName()
              + ".error.reload.failed");
          statusBar.setError(errorMsg);
        }
        else {
          Map<ClientProjectFile, String> result = getResult();
          if (result != null) {
            for (Entry<ClientProjectFile, String> entry : result.entrySet()) {
              LogPanelTextArea logPanelTextArea = panels.get(entry.getKey());
              logPanelTextArea.setText(entry.getValue());
            }
            scrollPane.revalidate();
          }
        }
      }
    };
    String title =
      LNG.get(ConsolidatedLogTab.class.getSimpleName() + ".title.reload.task",
        new Object[] { getTitle() });
    String message =
      LNG
      .get(ConsolidatedLogTab.class.getSimpleName() + ".message.reload.task");
    reloadTask.execute(getOwner(), title, message);
  }

  /**
   * L o contedo de um arquivos para uma String.
   *
   * @param file O arquivo a ser lido.
   *
   * @return O contedo do arquivo.
   */
  private String readFile(final ClientProjectFile file) {
    String content = "";
    try {
      final Client client = Client.getInstance();
      final Charset charset = client.getSystemDefaultCharset();
      content = TextReader.readAllWithoutTask(file, charset, null);
    }
    catch (Exception ex) {
      String errorMsg =
        LNG.get(ConsolidatedLogTab.class.getSimpleName()
          + ".error.log.read.failed", new Object[] { file.getName() });
      content = " [" + errorMsg + "]\n";
    }
    return content;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getMainComponent() {
    return mainPanel;
  }

  /**
   * Ao de recarga do contedo de todos os logs.
   */
  private final class ReloadAction extends AbstractAction {

    /**
     * Construtor.
     */
    private ReloadAction() {
      super(null, ApplicationImages.ICON_REFRESH_16);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      if (reloader.isRunning()) {
        reloader.stop();
      }
      else {
        reloader.start();
      }
    }
  }

  /**
   * Ao de exportao do contedo de todos os logs para um arquivo local.
   */
  private final class ExportAction extends AbstractAction {

    /**
     * Construtor.
     */
    private ExportAction() {
      super(null, ApplicationImages.ICON_EXPORT_16);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(ActionEvent e) {
      List<String> fileCodes = new ArrayList<String>();
      fileCodes.add("LOG");
      fileCodes.add("TEXT");
      String title =
        LNG.get(
          ConsolidatedLogTab.class.getSimpleName() + ".title.export.task",
          new Object[] { getTitle() });
      Window owner = getOwner();
      final ClientFile file =
        ClientLocalFileChooserUtil.browseSingleFileInSaveMode(owner, fileCodes,
          "LOG", title, true, lastSelectedDir);
      if (file != null) {
        lastSelectedDir = (ClientLocalFile) file.getParent();
        Task<Void> task = new ExportTask(file);
        String message =
          LNG.get(ConsolidatedLogTab.class.getSimpleName()
            + ".message.export.task");
        if (!task.execute(owner, title, message)) {
          String key =
            ConsolidatedLogTab.class.getSimpleName() + ".error.export.task";
          String error = LNG.get(key);
          statusBar.setError(error, 10);
        }
        else {
          String key =
            ConsolidatedLogTab.class.getSimpleName()
            + ".message.confirm.export.task";
          String success =
            LNG.get(key, new Object[] { "\'" + file.getName() + "\'" });
          statusBar.setInfo(success, 10);
        }
      }

    }
  }

  /**
   * Tarefa de exportao do contedo de todos os logs para um arquivo local.
   */
  private final class ExportTask extends RemoteTask<Void> {
    /**
     * Arquivo que vai receber o contedo consolidado dos logs.
     */
    private final ClientFile file;

    /**
     * Construtor.
     *
     * @param file arquivo que vai receber o contedo consolidado dos logs.
     */
    private ExportTask(ClientFile file) {
      this.file = file;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected void performTask() throws Exception {
      OutputStream outputStream = null;
      try {
        Set<ClientProjectFile> files =
          new TreeSet<ClientProjectFile>(panels.keySet());
        String fileContents = readFiles(files);
        outputStream = file.getOutputStream();
        outputStream.write(fileContents.getBytes());
      }
      catch (IOException ex) {
        ex.printStackTrace();
      }
      finally {
        if (outputStream != null) {
          try {
            outputStream.close();
          }
          catch (IOException ex) {
            ex.printStackTrace();
          }
        }
      }
    }

    /**
     * L e concatena o contedo de mltiplos arquivos em uma String.
     *
     * @param files Os arquivos a serem lidos.
     * @return O contedo dos arquivos.
     * @throws IOException em caso de falha de leitura ou escrita dos arquivos.
     */
    public String readFiles(final Set<ClientProjectFile> files)
      throws IOException {
      final StringWriter writer = new StringWriter();
      final PrintWriter printWriter = new PrintWriter(writer);
      if (files != null) {
        for (final ClientProjectFile clientFile : files) {
          String content = readFile(clientFile);
          if (content != null) {
            printWriter.print("\n");
            printWriter.printf("---- %s ----", clientFile.getName());
            printWriter.print("\n");
            printWriter.print(content);
          }
        }
      }
      return writer.getBuffer().toString();
    }
  }

}
