package csbase.client.util.sga;

import java.awt.Window;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.algorithms.tasks.CreateFlowConfiguratorTask;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.CommandSubmission;
import csbase.logic.CommonClientProject;
import csbase.logic.User;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * Tarefa de validao de um fluxo para execuo mltipla.
 */
public class CreateMultipleFlowSubmissionTask extends
  RemoteTask<MultipleCommandSubmission> {

  /**
   * Janela dona da task.
   */
  private Window owner;

  /**
   * O mapeamento dos parmetros para execuo de fluxos mltiplos.
   */
  private String[][] paramTable;

  /**
   * Configurador do fluxo a ser validado.
   */
  private FlowAlgorithmConfigurator configurator;

  /**
   * Construtor.
   *
   * @param owner a janela dona da task.
   * @param paramTable o mapeamento dos parmetros para execuo de fluxos
   *        mltiplos.
   * @param configurator o configurador do fluxo a ser validado.
   */
  public CreateMultipleFlowSubmissionTask(Window owner, String[][] paramTable,
    FlowAlgorithmConfigurator configurator) {
    this.owner = owner;
    this.paramTable = paramTable;
    this.configurator = configurator;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    FlowAlgorithmConfigurator flowConfigurator = configurator;
    MultipleCommandSubmission flowSubmission = new MultipleCommandSubmission();
    final Flow flow = flowConfigurator.getFlow();
    String flowName = flow.getName();
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    if (paramTable.length > 0) {
      for (int i = 1; i < paramTable.length; i++) {
        flow.setName(flowName + " #" + i);
        FlowAlgorithmConfigurator newConfigurator =
          CreateFlowConfiguratorTask.runTask(owner, flow);
        for (int j = 0; j < paramTable[i].length; j++) {
          String paramName = paramTable[0][j];
          String parameterValue = paramTable[i][j];
          try {
            newConfigurator.setParameterValue(paramName, parameterValue);
          }
          catch (ParameterNotFoundException e) {
            LocalizedMessage message =
              new LocalizedMessage(CreateMultipleFlowSubmissionTask.class,
                "parameter.not.found.error", new Object[] { paramName });
            ValidationError validationError = new ValidationError(message);
            flowSubmission.clearSubmissions();
            flowSubmission.setValidation(validationError);
            setResult(flowSubmission);
            return;
          }
          catch (ParseException e) {
            LocalizedMessage message =
              new LocalizedMessage(CreateMultipleFlowSubmissionTask.class,
                "parameter.parse.error", new Object[] { parameterValue,
                    paramName });
            ValidationError validationError = new ValidationError(message);
            flowSubmission.clearSubmissions();
            flowSubmission.setValidation(validationError);
            setResult(flowSubmission);
            return;
          }
        }

        Object projectId = project.getId();
        Object userId = User.getLoggedUser().getId();
        ValidationContext context =
          new ValidationContext(ValidationMode.FULL, projectId, userId);

        Validation result = newConfigurator.validate(context);
        if (!result.isWellSucceded()) {
          flowSubmission.clearSubmissions();
          flowSubmission.setValidation(result);
          setResult(flowSubmission);
          return;
        }
        String clientName = Client.getInstance().getClientName();
        CommandSubmission submission =
          new CommandSubmission(newConfigurator, project.getId(), clientName);
        flowSubmission.addSubmission(submission);
      }
    }
    setResult(flowSubmission);
  }

  /**
   * Mtodo de convenincia para chamada automtica da task.
   *
   * @param window janela dona da task.
   * @param paramTable o mapeamento dos parmetros para execuo de fluxos
   *        mltiplos.
   * @param configurator a configurador a ser validado.
   *
   * @return o resultado da validao.
   */
  final static public MultipleCommandSubmission runTask(Window window,
    String[][] paramTable, FlowAlgorithmConfigurator configurator) {
    CreateMultipleFlowSubmissionTask task =
      new CreateMultipleFlowSubmissionTask(window, paramTable, configurator);
    final String className =
      CreateMultipleFlowSubmissionTask.class.getSimpleName();
    final String title = LNG.get(className + ".title");
    final String msg = LNG.get(className + ".msg");
    if (task.execute(window, title, msg)) {
      return task.getResult();
    }
    else {
      LocalizedMessage errorMessage;
      if (task.wasCancelled()) {
        errorMessage =
          new LocalizedMessage(CreateMultipleFlowSubmissionTask.class,
            "cancelled");
      }
      else {
        Exception error = task.getError();
        error.printStackTrace();
        errorMessage =
          new LocalizedMessage(CreateMultipleFlowSubmissionTask.class, "error");
      }
      ValidationError validationError = new ValidationError(errorMessage);
      MultipleCommandSubmission submission = new MultipleCommandSubmission();
      submission.setValidation(validationError);
      return submission;
    }
  }

}
