package csbase.client.util.table;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

/**
 * Fbrica de Modelos de Clulas que utiliza Reflexo: ela cria modelos de
 * clulas atravs de reflexo. Ao criar a fbrica o usurio deve informar qual
 *  a classe do objeto armazenado na linha e do modelo de clulas a ser criado.
 * Ela procura um construtor que receba a classe do objeto armazenado na linha e
 * invoca este construtor na criao dos modelos de clulas.
 * 
 * @param <R> O tipo do valor armazenado na linha da tabela.
 */
public final class ReflectionCellModelFactory<R> implements CellModelFactory<R> {

  /**
   * O construtor de modelos de clulas.
   */
  private Constructor<? extends CellModel> cellModelClassConstrutor;

  /**
   * Cria a fbrica.
   * 
   * @param rowClass A classe do objeto armazenado na linha.
   * 
   * @param cellModelClass A classe do modelo de clulas.
   */
  public ReflectionCellModelFactory(Class<R> rowClass,
    Class<? extends CellModel> cellModelClass) {
    if (rowClass == null) {
      throw new IllegalArgumentException("O parmetro rowClass est nulo.");
    }
    if (cellModelClass == null) {
      throw new IllegalArgumentException(
        "O parmetro cellModelClass est nulo.");
    }

    Class<?> currentRowClass = rowClass;

    while (currentRowClass != null && this.cellModelClassConstrutor == null) {
      try {
        this.cellModelClassConstrutor = cellModelClass
          .getConstructor(currentRowClass);
      } catch (NoSuchMethodException e) {
      }
      currentRowClass = currentRowClass.getSuperclass();
    }

    if (this.cellModelClassConstrutor == null) {
      throw new IllegalArgumentException(
        String
          .format(
            "No foi encontrado um construtor adequado para criar modelos de clulas.\n"
              + "A classe deveria possuir pelo menos um construtor pblico que recebe "
              + "um nico argumento do tipo %s, ou de uma de suas superclasses.",
            rowClass.getName()));
    }
  }

  public CellModel create(R rowValue) {
    try {
      return cellModelClassConstrutor.newInstance(rowValue);
    } catch (InstantiationException e) {
      throw new IllegalStateException(e);
    } catch (IllegalAccessException e) {
      throw new IllegalStateException(e);
    } catch (InvocationTargetException e) {
      throw new IllegalStateException(e);
    }
  }
}
