/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.client.algorithms.parameters;

import java.awt.Color;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;

import javax.swing.BorderFactory;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import csbase.exception.ParseException;

/**
 * Campo de texto para capturar e formatar nmeros.
 *
 * @param <N> O tipo do nmero.
 *
 * @author lmoreira
 */
public abstract class NumberTextField<N extends Number> extends JTextField {
  /**
   * A borda-padro do campo de texto.
   */
  private Border defaultBorder;

  /**
   * A borda de erro.
   */
  private Border errorBorder;

  /**
   * O modelo.
   */
  private NumberTextFieldModel<N> model;

  /**
   * Cria o campo de texto.
   *
   * @param model O modelo (No aceita {@code null}).
   */
  protected NumberTextField(NumberTextFieldModel<N> model) {
    defaultBorder = getBorder();
    errorBorder = BorderFactory.createLineBorder(Color.RED, 2);
    setModel(model);
    updateView();
    getDocument().addDocumentListener(new DocumentListener() {

      @Override
      public void removeUpdate(DocumentEvent e) {
        validateText();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        validateText();
      }

      @Override
      public void changedUpdate(DocumentEvent e) {
        validateText();
      }
    });
    addFocusListener(new FocusListener() {

      @Override
      public void focusLost(FocusEvent e) {
        updateModel();
      }

      @Override
      public void focusGained(FocusEvent e) {
        // Ignora esse evento.
      }
    });
  }

  /**
   * Atualiza a viso com o valor do modelo.
   */
  public final void updateView() {
    N modelValue = model.getValue();
    if (modelValue == null) {
      setText("");
    }
    else if (!modelValue.equals(model.getErrorValue())) {
      setText(format(modelValue));
    }
  }

  /**
   * Formata o nmero.
   *
   * @param number O nmero (No aceita {@code null}).
   *
   * @return .
   */
  protected abstract String format(N number);

  /**
   * Cria o nmero representado no texto.
   *
   * @param text O texto (No aceita {@code null}).
   *
   * @return O nmero.
   *
   * @throws ParseException Se o texto no puder ser convertido para um nmero.
   */
  protected abstract N parse(String text) throws ParseException;

  /**
   * Atribui o modelo a este campo.
   *
   * @param model O modelo (No aceita {@code null}).
   */
  private void setModel(NumberTextFieldModel<N> model) {
    if (model == null) {
      throw new IllegalArgumentException("O parmetro model est nulo.");
    }
    this.model = model;
    this.model
      .addNumberTextFieldModelListener(new NumberTextFieldModelListener() {
        @Override
        public void wasChangedValue() {
          updateView();
        }
      });
  }

  /**
   * Atualiza o modelo com o valor da viso.
   */
  private void updateModel() {
    String text = getText();
    if (text.length() == 0) {
      model.setValue(null);
    }
    else {
      try {
        N number = parse(text);
        model.setValue(number);
      }
      catch (ParseException exception) {
        N err = model.getErrorValue();
        model.setValue(err);
      }
    }
  }

  /**
   * Valida o valor da viso.
   */
  private void validateText() {
    String text = getText();
    if (text.length() == 0) {
      setBorder(defaultBorder);
    }
    else {
      try {
        parse(text);
        setBorder(defaultBorder);
      }
      catch (ParseException exception) {
        setBorder(errorBorder);
      }
    }
  }
}
