package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Action;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.algorithmsmanager.actions.AlgorithmCreateAction;
import csbase.client.applications.algorithmsmanager.actions.AlgorithmManagementAction;
import csbase.client.applications.algorithmsmanager.actions.AlgorithmRemoveAction;
import csbase.client.applications.algorithmsmanager.actions.CommonManagementAction;
import csbase.client.applications.algorithmsmanager.actions.ListOutdatedAlgorithmAction;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.logic.algorithms.AlgorithmInfo;

/**
 * Classe que define o painel para seleo de dados e para as operaes que
 * podem ser realizadas sobre esses dados selecionados, na funcionalidade de
 * gerenciamento de algoritmos.
 *
 * A seleo de algoritmos  feita atravs da seleo de um n em uma lista de
 * algoritmos. As operaes bsicas definidas para atuar sobre algoritmos, so
 * as aes de criao de um novo algoritmo e a de remoo de um ou mais
 * algoritmos.
 *
 */
public class AlgorithmSelectionPanel extends CommonSelectionPanel {

  /**
   * Viso da lista de algoritmos
   */
  private AlgorithmListView algorithmListView;

  /**
   * Nmero do item algoritmo selecionado correntemente na lista
   */
  private int currentAlgorithmRow;

  /** Ao para criao de um novo algoritmo */
  private AlgorithmCreateAction algorithmCreateAction;

  /** Ao para remoo de um ou mais algoritmos selecionados */
  private AlgorithmRemoveAction algorithmRemoveAction;

  /** Ao para remoo de um ou mais algoritmos selecionados */
  private ListOutdatedAlgorithmAction listOutdatedAlgorithmAction;

  /** Painel principal de seleo de algoritmos */
  private JPanel mainSelectionPanel;

  /** Painel com a lista de algoritmos */
  private JPanel algoListPanel;

  /**
   * Indica que as informaes j foram atualizadas. Foi necessrio para tratar
   * os casos de seleo de um item da JList em que o evento de mudana chega,
   * mas  somente para o item aparecer selecionado e no precisa inicializar
   * esse elemento
   */
  private boolean infoAlreadyUpdated;

  /**
   * Indica que est sendo feita uma seleo inicial na JList.
   */
  private boolean isInitialSelection;

  /**
   * Constri o painel de seleo de algoritmos.
   *
   * @param action ao de gerenciamento de algoritmos que criou esse painel
   */
  public AlgorithmSelectionPanel(CommonManagementAction action) {
    super(action);
  }

  @Override
  public AlgorithmManagementAction getManagementAction() {
    return (AlgorithmManagementAction) super.getManagementAction();
  }

  @Override
  protected JPanel buildMainSelectionPanel() {
    mainSelectionPanel = new JPanel(new GridBagLayout());
    algoListPanel = getAlgorithmListPanel();
    mainSelectionPanel.add(algoListPanel,
      new GBC(0, 0).both().west().insets(0, 0, 0, 0));
    return mainSelectionPanel;
  }

  @Override
  protected List<Action> buildOperationActions() {
    List<Action> actions = new ArrayList<Action>();
    algorithmCreateAction = new AlgorithmCreateAction(this, null);
    algorithmRemoveAction = new AlgorithmRemoveAction(this, null);
    algorithmRemoveAction.setEnabled(false);
    actions.add(algorithmCreateAction);
    actions.add(algorithmRemoveAction);
    return actions;
  }

  @Override
  public List<DataInterface> getSelectedDataList() {
    List<DataInterface> selectedDataList = new ArrayList<DataInterface>();
    if (algorithmListView != null) {
      selectedDataList = algorithmListView.getSelectedDataList();
    }
    return selectedDataList;
  }

  @Override
  protected boolean confirmSelectionChanged() {
    boolean confirmed = false;
    if (getManagementAction().getEditPanel() != null
      && getManagementAction().getEditPanel().confirmDataChanged()) {
      currentAlgorithmRow = getListSelectedRow();
      confirmed = true;
    }
    infoAlreadyUpdated = true;
    algorithmListView.selectAlgorithmIndex(currentAlgorithmRow);
    infoAlreadyUpdated = false;
    return confirmed;
  }

  /**
   * Cria o painel com a lista de algoritmos.
   *
   * @return o painel com a lista de algoritmos
   */
  private JPanel getAlgorithmListPanel() {
    algorithmListView =
      new AlgorithmListView(this, getApplication().getAllAgorithms());
    return algorithmListView.getPanel();
  }

  /**
   * Inicializa a seleo de um novo algoritmo na lista de algoritmos. O painel
   * de edio precisa ser atualizado na gerncia de algoritmos.
   *
   * @param item item algoritmo selecionado na lista
   */
  protected void initAlgorithmSelection(AlgorithmListItem item) {
    if (infoAlreadyUpdated) {
      return;
    }
    if (isCreationEditionPanel()) {
      if (!confirmSelectionChanged()) {
        return;
      }
      else {
        getManagementAction().updateEditPanel(true);
        ((AlgorithmEditDataPanel) getManagementAction().getEditPanel())
        .changeAlgorithmItemEdition(item);
        verifyAndChangeButtonsState();
      }
    }

    if (!isInitialSelection) {
      if (changedSelectedAlgorithm()) {
        // O dado selecionado mudou
        if (!confirmSelectionChanged()) {
          return;
        }
      }
    }
    getManagementAction().updateEditPanel(true);
    ((AlgorithmEditDataPanel) getManagementAction().getEditPanel())
    .changeAlgorithmItemEdition(item);
    verifyAndChangeButtonsState();

  }

  /**
   * Verifica se o algoritmo selecinado mudou em relao ao algoritmo corrente.
   *
   * @return retorna true se o algoritmo selecionado for diferente do que estava
   *         correntemente selecionado, caso contrrio, retorna false
   */
  private boolean changedSelectedAlgorithm() {
    return algorithmListView.getSelectedIndex() != currentAlgorithmRow;
  }

  /**
   * Verifica se o painel de edio dos dados se refere  criao do dado.
   *
   * @return retorna true se o painel de edio dos dados se refere  criao do
   *         dado, caso contrrio, retorna false
   */
  private boolean isCreationEditionPanel() {
    return getManagementAction().getEditPanel() instanceof AlgorithmCreateDataPanel;
  }

  /**
   * Obtm o algoritmo selecionado na lista.
   *
   * @return retorna o algoritmo selecionad na lista de dados, caso contrrio,
   *         retorna null
   */
  public AlgorithmListItem getSelectedAlgorithm() {
    return (AlgorithmListItem) super.getSelectedData();
  }

  /**
   * Obtm o nmero do n selecionado na lista de algoritmos.
   *
   * @return o nmero do n selecionado na lista de algoritmos, caso contrrio,
   *         retorna -1
   */
  private int getListSelectedRow() {
    if (algorithmListView != null) {
      return algorithmListView.getSelectedIndex();
    }
    return -1;
  }

  /**
   * Verifica e modifica o estado das aes sobre um ou mais algoritmos de
   * acordo com as selees feitas pelo usurio na lista de algoritmos.
   *
   */
  private void verifyAndChangeButtonsState() {
    // ao de Criar algoritmo
    boolean enableAddButton = false;
    enableAddButton =
      (algorithmListView.getSelectionCount() <= 1) ? true : false;
    enableAction(algorithmCreateAction, enableAddButton);

    // ao de Remover algoritmo(s)
    boolean enableRemoveButton = false;
    enableRemoveButton =
      (algorithmListView.getSelectionCount() > 0) ? true : false;
    enableAction(algorithmRemoveAction, enableRemoveButton);
  }

  /**
   * Seleciona um algoritmo na lista de algoritmos.
   *
   * @param algoInfo algoritmo a ser selecionado
   */
  public void selectAlgorithm(AlgorithmInfo algoInfo) {
    AlgorithmListItem algorithmItem =
      algorithmListView.getAlgorithmItem(algoInfo);
    algorithmListView.selectAlgorithm(algoInfo);
    if (!infoAlreadyUpdated) {
      initAlgorithmSelection(algorithmItem);
    }
  }

  /**
   * Seleciona o algoritmo que est no ndice especificado.
   *
   * @param index ndice a ser selecionado na lista
   */
  private void selectAlgorithmIndex(int index) {
    if (index >= 0) {
      algorithmListView.selectAlgorithmIndex(index);
    }
  }

  @Override
  protected void handleAlgorithmCreated(AlgorithmInfo algoInfo) {
    updateSelectionPanel();
    selectAlgorithm(algoInfo);
    // Posicionar na aba de verses aps a criao do algoritmo
    CommonInfoEditPanel versionPanel =
      ((AlgorithmEditDataPanel) getManagementAction().getEditPanel())
      .getVersionPanel();
    getManagementAction().getEditPanel().setSelectedInfoPanel(versionPanel);
  }

  @Override
  protected void handleAlgorithmRemoved(AlgorithmInfo algoInfo) {
    updateSelectionPanel();
    selectAlgorithmIndex((currentAlgorithmRow < algorithmListView.getCount()) ? currentAlgorithmRow
      : --currentAlgorithmRow);
  }

  @Override
  protected void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
    updateSelectionPanel();
    infoAlreadyUpdated = true;
    selectAlgorithm(algoInfo);
    infoAlreadyUpdated = false;
  }

  /**
   * Atualiza o painel de seleo a partir do evento de algoritmo ocorrido. Esse
   * mtodo reconstri o painel de seleo, reconstruindo a lista de algoritmos.
   *
   */
  private void updateSelectionPanel() {
    mainSelectionPanel.remove(algoListPanel);
    algoListPanel = getAlgorithmListPanel();
    mainSelectionPanel.add(algoListPanel,
      new GBC(0, 0).both().west().insets(0, 0, 0, 0));
    mainSelectionPanel.revalidate();
  }

  /**
   * Seleciona um algoritmo inicial a ser exibido.
   */
  public void selectInitialData() {
    isInitialSelection = true;
    selectAlgorithmIndex(0);
    isInitialSelection = false;
  }
}
