package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Action;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.tree.FilterableTreePanel;
import csbase.client.applications.algorithmsmanager.actions.CategoryCreateAction;
import csbase.client.applications.algorithmsmanager.actions.CategoryManagementAction;
import csbase.client.applications.algorithmsmanager.actions.CategoryRemoveAction;
import csbase.client.applications.algorithmsmanager.models.CategoryNodeInterface;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;

/**
 * Classe que define o painel para seleo de dados e para as operaes que
 * podem ser realizadas sobre esses dados selecionados, na funcionalidade de
 * gerenciamento de categorias.
 * 
 * A seleo de categorias  feita atravs da seleo de um n em uma rvore de
 * categorias. As operaes bsicas definidas para atuar sobre categorias, so
 * as aes de criao de uma nova categoria e a de remoo de uma ou mais
 * categorias.
 * 
 */
public class CategorySelectionPanel extends CommonSelectionPanel {
  /**
   * Viso da rvore de categorias, a partir da qual se obtm a rvore com
   * filtro
   */
  private CategoryTreeView categoryTreeView;

  /** Ao para criao de uma nova categoria */
  private CategoryCreateAction categoryCreateAction;

  /** Ao para remoo de uma ou mais categorias selecionadas */
  private CategoryRemoveAction categoryRemoveAction;

  /** Painel principal de seleo */
  private JPanel mainSelectionPanel;

  /** Painel com a rvore filtrvel pelo nome do n */
  private FilterableTreePanel dataTreePanel;

  /**
   * Constri o painel de seleo de categorias.
   * 
   * @param action ao de gerenciamento de categorias que criou esse painel
   */
  public CategorySelectionPanel(CategoryManagementAction action) {
    super(action);
  }

  @Override
  public CategoryManagementAction getManagementAction() {
    return (CategoryManagementAction) super.getManagementAction();
  }

  @Override
  protected JPanel buildMainSelectionPanel() {
    mainSelectionPanel = new JPanel(new GridBagLayout());
    dataTreePanel = getCategoryTreePanel();
    mainSelectionPanel.add(dataTreePanel, new GBC(0, 0).both().west().insets(5,
      5, 5, 5));
    return mainSelectionPanel;
  }

  @Override
  protected List<Action> buildOperationActions() {
    List<Action> actions = new ArrayList<Action>();
    categoryCreateAction = new CategoryCreateAction(this, null);
    categoryRemoveAction = new CategoryRemoveAction(this, null);
    actions.add(categoryCreateAction);
    actions.add(categoryRemoveAction);
    return actions;
  }

  @Override
  public List<DataInterface> getSelectedDataList() {
    List<DataInterface> selectedDataList = new ArrayList<DataInterface>();
    if (categoryTreeView != null) {
      selectedDataList = categoryTreeView.getSelectedDataList();
    }
    return selectedDataList;
  }

  /**
   * Obtm a categoria selecionada na rvore.
   * 
   * @return retornaa categoria selecionada na rvore de dados, caso contrrio,
   *         retorna null
   */
  public CategoryNodeInterface getSelectedCategory() {
    return (CategoryNodeInterface) getSelectedData();
  }

  @Override
  protected boolean confirmSelectionChanged() {
    boolean confirmed = false;
    if (getManagementAction().getEditPanel() != null
      && getManagementAction().getEditPanel().confirmDataChanged()) {
      confirmed = true;
    }
    return confirmed;
  }

  /**
   * Cria o painel com filtro da rvore de categorias.
   * 
   * @return o painel com filtro criado
   */
  private FilterableTreePanel getCategoryTreePanel() {
    categoryTreeView = new CategoryTreeView(this, getApplication()
      .getAllCategories(false));
    FilterableTreePanel dataTreePanel = categoryTreeView.getTreePanel();
    return dataTreePanel;
  }

  /**
   * Obtm a categoria selecionada na rvore de dados.
   * 
   * @return retorna a categoria selecionada na rvore de dados. Se a seleo
   *         atual no corresponder a uma categoria ou no tiver nenhum item
   *         selecionado, retorna null
   */
  public CategoryNodeInterface getTreeSelectedCategory() {
    if (categoryTreeView != null) {
      return categoryTreeView.getFirstSelectedNode();
    }
    return null;
  }

  /**
   * Verifica e modifica o estado das aes sobre uma ou mais categorias de
   * acordo com as selees feitas pelo usurio na rvore de categorias.
   * 
   */
  public void verifyAndChangeButtonsState() {
    // ao de Criar categoria
    boolean enableAddButton = false;
    enableAddButton = (categoryTreeView.getSelectionCount() <= 1) ? true
      : false;
    categoryCreateAction.setEnabled(enableAddButton);

    // ao de Remover categoria
    boolean enableRemoveButton = false;
    if (!categoryTreeView.isRootNodeSelected()) {
      enableRemoveButton = (categoryTreeView.getSelectionCount() > 0) ? true
        : false;
    }
    categoryRemoveAction.setEnabled(enableRemoveButton);

  }

  /**
   * Seleciona o n raiz da rvore de dados.
   * 
   */
  public void selectRootNode() {
    categoryTreeView.selectRootNode();
  }

  /**
   * Seleciona uma determinada categoria na rvore de categorias.
   * 
   * @param category categoria a ser selecionada
   */
  private void selectCategory(Category category) {
    categoryTreeView.selectCategory(category);
  }

  /**
   * Seleciona a categoria corrente, inicializando seus valores.
   */
  public void selectCurrentCategory() {
    categoryTreeView.selectCurrentCategory();
  }

  @Override
  protected void handleCategoryCreated(Category category) {
    updateSelectionPanel();
    selectCategory(category);
  }

  @Override
  protected void handleCategoryRemoved(Category category) {
    updateSelectionPanel();
    selectCurrentCategory();
  }

  @Override
  protected void handleCategoryUpdated(CategorySet modifiedCategorySet) {
    updateSelectionPanel();
    selectCurrentCategory();
    // TODO SUBSTITUIR
    // selectCategory(modifiedCategorySet.getFirstCategory());
  }

  /**
   * Atualiza o painel de seleo a partir do evento de categoria ocorrido. Esse
   * mtodo reconstri o painel de seleo, reconstruindo a rvore de
   * categorias.
   */
  private void updateSelectionPanel() {
    mainSelectionPanel.remove(dataTreePanel);
    dataTreePanel = updateCategoryTreePanel();
    mainSelectionPanel.add(dataTreePanel, new GBC(0, 0).both().west().insets(5,
      5, 5, 5));
    mainSelectionPanel.revalidate();
    getApplication().getApplicationFrame().repaint();
  }

  /**
   * Atualiza a rvore de categorias, mantendo a linha atualmente selecionada.
   * 
   * @return o painel com filtro que contm a nova rvore
   */
  private FilterableTreePanel updateCategoryTreePanel() {
    categoryTreeView.updateCategoryTree(getApplication()
      .getAllCategories(false));
    dataTreePanel = categoryTreeView.getTreePanel();
    return dataTreePanel;
  }

  /**
   * Inicializa a edio do n de categoria especificado.
   * 
   * @param categoryNode n categoria a ser editado
   */
  public void initCategoryNodeEdition(CategoryNodeInterface categoryNode) {
    if (categoryNode.isRootNode()) {
      getManagementAction().updateEditPanel(false);
    }
    else {
      getManagementAction().updateEditPanel(true);
      ((CategoryEditDataPanel) getManagementAction().getEditPanel())
        .changeUpdateInfo(categoryNode);
    }
  }

  /**
   * Verifica se o painel de edio dos dados se refere  criao do dado.
   * 
   * @return retorna true se o painel de edio dos dados se refere  criao do
   *         dado, caso contrrio, retorna false
   */
  public boolean isCreationEditionPanel() {
    return getManagementAction().getEditPanel() instanceof CategoryCreateDataPanel;
  }

}
