package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.List;
import java.util.SortedSet;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.DataPropertiesListener;
import csbase.client.applications.algorithmsmanager.models.DataInterface;

/**
 * Painel para associar os dados de acordo com o modo de seleo da rvore de
 * dados (categorias ou algoritmos).
 * 
 * Este componente vai permitir associar ou desassociar algoritmos de categorias
 * e vice-versa.
 * 
 */
public class DataBindingPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> {

  /**
   * O componente que contm os elementos a serem selecionados.
   */
  private ContainerSelection<DataInterface> containerSelection;

  /** Dados disponveis para serem associados */
  private SortedSet<DataInterface> availableData;

  /** Dados previamente selecionados como associados */
  private SortedSet<DataInterface> previousSelectedData;

  /** Texto que descreve o tipo de dados a ser associado */
  private String description;

  /** Listeners da associao entre algoritmos e categorias */
  private List<DataPropertiesListener> bindListeners;

  /**
   * Constri o painel.
   * 
   * @param application aplicao de gerncia de algoritmos
   * @param availableData dados disponveis para serem associados
   * @param selectedData dados selecionados para serem associados
   * @param description Texto que descreve o tipo de dados a ser associado
   */
  DataBindingPanel(AlgorithmsManager application,
    SortedSet<DataInterface> availableData,
    SortedSet<DataInterface> selectedData, String description) {
    super(application);
    this.availableData = availableData;
    this.previousSelectedData = selectedData;
    this.description = description;
    this.bindListeners = new Vector<DataPropertiesListener>();
    buildPanel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());
    setBorder(BorderFactory.createEtchedBorder());
    JLabel descriptionLabel = new JLabel(description);

    add(descriptionLabel, new GBC(0, 0).horizontal().west().insets(5, 5, 5, 5));
    add(getDataSelectionPanel(getAvailableItems(), previousSelectedData),
      new GBC(0, 1).both().west().insets(5, 5, 5, 5));
  }

  /**
   * Cria o painel para seleo dos dados.
   * 
   * @param availableItems dados disponveis para adio
   * @param selectedItems dados selecionados no grupo
   * 
   * @return o painel criado
   */
  private JPanel getDataSelectionPanel(SortedSet<DataInterface> availableItems,
    SortedSet<DataInterface> selectedItems) {
    ObjectTableProvider selectableProvider =
      new MyObjectTableProvider(getString("DataBindingPanel.available_items"));
    ObjectTableProvider selectedProvider =
      new MyObjectTableProvider(getString("DataBindingPanel.selected_items"));

    containerSelection =
      new ContainerSelection<DataInterface>(selectableProvider,
        selectedProvider, true, false);
    containerSelection.loadItems(availableItems, selectedItems);
    containerSelection.adjustTableColumns();
    containerSelection
      .addContainerSelectionListener(new ContainerSelectionListener() {

        @Override
        public void containerChanged() {
          notifyListeners();
        }
      });
    return containerSelection.getPanel();
  }

  /**
   * Obtm os itens de dados disponveis para seleo.
   * 
   * @return a lista contendo os itens disponveis para seleo
   */
  private SortedSet<DataInterface> getAvailableItems() {
    return availableData;
  }

  /**
   * Obtm os itens de dados selecionados para seleo.
   * 
   * @return a lista contendo os itens selecionados para seleo
   */
  public List<DataInterface> getSelectedItems() {
    return containerSelection.getSelectedItems();
  }

  /**
   * Utilizada para fornecer dados sobre o tem de enumerao para as tabelas do
   * container de seleo.
   */
  private final class MyObjectTableProvider extends DefaultObjectTableProvider {
    /**
     * O nome da coluna.
     */
    private String columnName;

    /**
     * @param columnName O nome da coluna (No aceita {@code null}).
     */
    MyObjectTableProvider(String columnName) {
      this.columnName = columnName;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getColumnNames() {
      return new String[] { this.columnName };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getColumnClasses() {
      return new Class[] { String.class };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object[] getCellValues(Object row) {
      DataInterface item = (DataInterface) row;
      return new Object[] { item.getFullName() };
    }
  }

  /**
   * Altera os itens do painel de associao entre algoritmos e categorias.
   * 
   * @param availableItems itens disponveis
   * @param selectedItems itens selecionados
   */
  public void updateItems(SortedSet<DataInterface> availableItems,
    SortedSet<DataInterface> selectedItems) {
    if (containerSelection != null) {
      containerSelection.loadItems(availableItems, selectedItems);
      containerSelection.adjustTableColumns();
    }
    revalidate();
  }

  /**
   * Notifica os listeners de que ocorreram mudanas na associao entre
   * algoritmos e categorias
   * 
   */
  protected void notifyListeners() {
    for (DataPropertiesListener bindListener : bindListeners) {
      bindListener.propertiesChanged(true);
    }
  }

  /**
   * Adiciona um listener de mudanas na associao entre algoritmos e
   * categorias.
   * 
   * @param listener listener de mudanas na associao entre algoritmos e
   *        categorias
   */
  public void addDataBindListener(DataPropertiesListener listener) {
    bindListeners.add(listener);
  }
}
