package csbase.client.applications.filetransfer;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIResources;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTree;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.gui.MultiImageButton;

/**
 * A classe {@link FileTransfer} implementa o dilogo de cpia de arquivos entre
 * projetos. No momento, as combos s mostram projetos do usurio corrente e
 * {@link ProjectComboBox} no est sendo notificada de incluso/excluso do
 * usurio em algum projeto de outro usurio.
 * 
 * @author Ana Lodi
 */
public class FileTransfer extends Application {

  /** Painel com o seletor de projeto e arquivo origem */
  private ProjectFileSelectorPanel sourceFilePanel;

  /** Painel com o seletor de projeto e arquivo destino */
  private ProjectFileSelectorPanel targetFilePanel;

  /**
   * Construtor.
   * 
   * @param id identificador da aplicao.
   */
  public FileTransfer(String id) {
    super(id);
    createMainFrame();
    showFrame();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    sourceFilePanel.removeObservers();
    targetFilePanel.removeObservers();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userCanKillApplication() {
    return true;
  }

  /** Exibe a janela. */
  private void showFrame() {
    getApplicationFrame().setSize(new Dimension(800, 600));
    getApplicationFrame().center(DesktopFrame.getInstance().getDesktopFrame());
    getApplicationFrame().setVisible(true);
  }

  /** Cria a janela principal. */
  private void createMainFrame() {
    Container cp = getApplicationFrame().getContentPane();
    cp.setLayout(new GridBagLayout());
    cp.add(createTextPanel(),
      new GBC(0, 0).horizontal().north().insets(10, 10, 0, 10));
    cp.add(createTransferPanel(),
      new GBC(0, 1).both().center().insets(10, 11, 0, 10));
    cp.add(createButtonPanel(),
      new GBC(0, 2).horizontal().center().insets(10, 10, 5, 10));
  }

  /**
   * Cria o painel de Transferncia
   * 
   * @return painel.
   */
  private JPanel createTransferPanel() {
    JPanel tp = new JPanel(new GridBagLayout());
    try {
      sourceFilePanel =
        new ProjectFileSelectorPanel(this, true,
          getString("project.source.label"), false);
      tp.add(sourceFilePanel,
        new GBC(0, 0).both().northwest().insets(0, 0, 0, 0));
      tp.add(createActionButtonsPanel(),
        new GBC(1, 0).fillx().center().insets(0, 10, 0, 10));
      targetFilePanel =
        new ProjectFileSelectorPanel(this, false,
          getString("project.target.label"), false);
      tp.add(targetFilePanel,
        new GBC(2, 0).both().northeast().insets(0, 0, 0, 0));

    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(getApplicationFrame(),
        getApplicationFrame().getTitle(), e);
    }
    return tp;
  }

  /**
   * Cria o painel com o boto de Fechar
   * 
   * @return painel.
   */
  private JPanel createButtonPanel() {
    JButton exitButton = new JButton(new ApplicationExitAction(this));
    JPanel bp = new JPanel(new FlowLayout());
    bp.add(exitButton);
    return bp;
  }

  /**
   * Cria o Painel com o Texto explicativo
   * 
   * @return painel.
   */
  private JPanel createTextPanel() {
    JPanel tp = new JPanel(new GridBagLayout());
    tp.setForeground(Color.GRAY);
    JLabel icon = new JLabel(ApplicationImages.ICON_INFORMATION_32);
    JLabel lb1 = new JLabel(getString("move.text"));
    JLabel lb2 = new JLabel(getString("copy.text"));
    lb1.setForeground(Color.GRAY);
    lb2.setForeground(Color.GRAY);
    tp.add(icon, new GBC(0, 0).height(2).none().insets(0, 0, 0, 5));
    tp.add(lb1, new GBC(1, 0).horizontal().northwest());
    tp.add(lb2, new GBC(1, 1).horizontal().northwest());
    return tp;
  }

  /**
   * Cria o painel de botes para transferncia de arquivos.
   * 
   * @return painel.
   */
  private JComponent createActionButtonsPanel() {
    JPanel buttonsPanel = new JPanel(new GridBagLayout());

    MultiImageButton moveRightButton =
      new MultiImageButton(null, getString("button.move"),
        GUIResources.SELECTOR_ADD_ICON);
    moveRightButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        moveFiles(sourceFilePanel.getProjectTree(),
          targetFilePanel.getProjectTree());
      }
    });

    MultiImageButton copyRightButton =
      new MultiImageButton(null, getString("button.copy"),
        GUIResources.SELECTOR_ADD_ICON);
    copyRightButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        copyFiles(sourceFilePanel.getProjectTree(),
          targetFilePanel.getProjectTree());
      }
    });

    MultiImageButton moveLeftButton =
      new MultiImageButton(GUIResources.SELECTOR_REMOVE_ICON,
        getString("button.move"), null);
    moveLeftButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        moveFiles(targetFilePanel.getProjectTree(),
          sourceFilePanel.getProjectTree());
      }
    });

    MultiImageButton copyLeftButton =
      new MultiImageButton(GUIResources.SELECTOR_REMOVE_ICON,
        getString("button.copy"), null);
    copyLeftButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        copyFiles(targetFilePanel.getProjectTree(),
          sourceFilePanel.getProjectTree());
      }
    });

    buttonsPanel.add(moveRightButton, new GBC(0, 0).fillx().center());
    buttonsPanel.add(copyRightButton,
      new GBC(0, 1).fillx().center().insets(5, 0, 0, 0));
    buttonsPanel.add(moveLeftButton,
      new GBC(0, 2).fillx().center().insets(30, 0, 0, 0));
    buttonsPanel.add(copyLeftButton,
      new GBC(0, 3).fillx().center().insets(5, 0, 0, 0));
    return buttonsPanel;
  }

  /**
   * Realiza a cpia dos arquivos, caso hajam arquivos selecionados no projeto
   * de origem e um diretrio selecionado no projeto de destino.
   * 
   * @param source Projeto de origem.
   * @param target Projeto de destino.
   */
  private void copyFiles(ProjectTree source, ProjectTree target) {
    if (canTransfer(source, target)) {
      source.startCopyAction();
      target.startPasteAction();
    }
  }

  /**
   * Realiza a movimentao dos arquivos, caso hajam arquivos selecionados no
   * projeto de origem e um diretrio selecionado no projeto de destino.
   * 
   * @param source Projeto de origem.
   * @param target Projeto de destino.
   */
  private void moveFiles(ProjectTree source, ProjectTree target) {

    if (canTransfer(source, target)) {
      source.startCutAction();
      target.startPasteAction();
    }
  }

  /**
   * Valida se  possvel iniciar a transferncia de arquivos, ou seja, se
   * existem arquivos selecionados no projeto de origem e um diretrio
   * selecionado no projeto de destino.
   * 
   * @param source Projeto de origem.
   * @param target Projeto de destino.
   * 
   * @return True, se existem arquivos selecionados no projeto de origem e um
   *         diretrio selecionado no projeto de destino. False, caso qualquer
   *         uma das condies abaixo seja verdadeira: <br>
   *         - Nenhum arquivo de origem selecionado. <br>
   *         - Nenhum diretrio de destino selecionado. <br>
   *         - Mais de um diretrio de destino selecionado. <br>
   *         - Arquivo selecionado como destino.
   */
  private boolean canTransfer(ProjectTree source, ProjectTree target) {

    if (source.getSelectionCount() == 0) {
      // Nenhum arquivo de origem selecionado.
      StandardErrorDialogs.showErrorDialog(getApplicationFrame(),
        getApplicationFrame().getTitle(), getString("missing.source.msg"));
      return false;
    }
    if (target.getSelectionCount() == 1) {
      // Apenas um destino selecionado.
      if (target.getSelectedFile().isDirectory()) {
        // Destino selecionado  um diretrio.
        return true;
      }
    }
    StandardErrorDialogs.showErrorDialog(getApplicationFrame(),
      getApplicationFrame().getTitle(), getString("missing.target.msg"));
    return false;
  }
}