package csbase.client.applications.pdfviewer;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.io.IOException;
import java.io.InputStream;

import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JToolBar;

import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.pdfviewer.core.PDFCorePanel;
import tecgraf.javautils.pdfviewer.core.PDFCorePanel.ZoomFitPolicy;
import tecgraf.javautils.pdfviewer.core.PDFDocument;
import tecgraf.javautils.pdfviewer.core.PDFReader;
import tecgraf.javautils.pdfviewer.core.listeners.PDFDocumentOpenCloseListener;
import tecgraf.javautils.pdfviewer.core.listeners.PDFZoomFitPolicyChangedListener;
import tecgraf.javautils.pdfviewer.viewer.PDFViewerPanel;
import tecgraf.javautils.pdfviewer.viewer.listeners.PDFThumbnailsVisibilityListener;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.pdfviewer.actions.OpenLocalPdfFileAction;
import csbase.client.applications.pdfviewer.actions.OpenProjectPdfFileAction;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.project.ProjectTreeListener;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Aplicao para visualizar arquivos PDF.
 * 
 * @author Tecgraf
 */
public class PDFViewer extends Application {

  /**
   * PdfViewerPanel principal
   */
  final private PDFViewerPanel pdfViewerPanel;

  /**
   * Ao que abre um arquivo local
   */
  final private Action openLocalFileAction = new OpenLocalPdfFileAction(this);

  /**
   * Ao que abre um arquivo da rea de projetos
   */
  final private Action openProjFileAction = new OpenProjectPdfFileAction(this);

  /**
   * habilita ou desabilita a ao de abrir arquivo do projeto de acordo com
   * mudanas na rea de projeto do cliente (se ele abriu ou fechou um projeto)
   */
  private final ProjectTreeListener openProjFileEnabler =
    new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject project) {
        openProjFileAction.setEnabled(project != null);
      }
    };

  /**
   * Construtor obrigatrio
   * 
   * @param id identificador
   */
  public PDFViewer(String id) {
    super(id);
    getApplicationFrame().setPreferredSize(new Dimension(800, 600));
    getApplicationFrame().setLayout(new BorderLayout());
    pdfViewerPanel = new PDFViewerPanel(getLocale());
    getApplicationFrame().add(pdfViewerPanel, BorderLayout.CENTER);

    setupToolBar();
    setupMenuBar();
    addListeners();
    getApplicationFrame().pack();
  }

  /**
   * Adiciona os listeners desta camada da aplicao
   */
  private void addListeners() {
    addOpenProjFileEnablerListener();
    pdfViewerPanel
      .addPDFDocumentOpenCloseListener(new PDFDocumentOpenCloseListener() {
        @Override
        public void documentOpened(PDFCorePanel pdfViewerPanel) {
          // no faz nada, estamos interessados na ao de fechar
        }

        @Override
        public void documentClosed(PDFCorePanel source) {
          setTitleFilename(null);
        }
      });

  }

  /**
   * Monta a barra de menus
   */
  private void setupMenuBar() {
    JMenu fileMenu = new JMenu(getString("file.menu"));

    JMenuItem openProjFileMenuItem = new JMenuItem(openProjFileAction);
    fileMenu.add(openProjFileMenuItem);

    if (canOpenLocalFiles()) {
      JMenuItem openLocalFileMenuItem = new JMenuItem(openLocalFileAction);
      fileMenu.add(openLocalFileMenuItem);
    }

    fileMenu.addSeparator();
    fileMenu.add(pdfViewerPanel.getPDFCloseAction());
    fileMenu.addSeparator();
    fileMenu.add(new ApplicationExitAction(this));

    JMenu viewMenu = new JMenu(getString("view.menu"));
    viewMenu.add(pdfViewerPanel.getPrevPageAction());
    viewMenu.add(pdfViewerPanel.getNextPageAction());
    viewMenu.addSeparator();

    viewMenu.add(pdfViewerPanel.getZoomInAction());
    viewMenu.add(pdfViewerPanel.getZoomOutAction());

    final ButtonGroup fitButtonGroup = new ButtonGroup();

    final JCheckBoxMenuItem fitWholePageMenuItem =
      new JCheckBoxMenuItem(pdfViewerPanel.getFitWholePageAction());
    fitButtonGroup.add(fitWholePageMenuItem);
    viewMenu.add(fitWholePageMenuItem);
    final JCheckBoxMenuItem fitWidthMenuItem =
      new JCheckBoxMenuItem(pdfViewerPanel.getFitWidthAction());
    fitButtonGroup.add(fitWidthMenuItem);
    viewMenu.add(fitWidthMenuItem);
    final JCheckBoxMenuItem fitHeightMenuItem =
      new JCheckBoxMenuItem(pdfViewerPanel.getFitHeightAction());
    fitButtonGroup.add(fitHeightMenuItem);
    viewMenu.add(fitHeightMenuItem);

    pdfViewerPanel
      .addPDFZoomFitPolicyChangedListener(new PDFZoomFitPolicyChangedListener() {

        @Override
        public void zoomFitPolicyChanged(ZoomFitPolicy policy) {
          if (policy == ZoomFitPolicy.FREE) {
            fitButtonGroup.clearSelection();
          }
          else if (policy == ZoomFitPolicy.WIDTH) {
            fitWidthMenuItem.setSelected(true);
          }
          else if (policy == ZoomFitPolicy.HEIGHT) {
            fitHeightMenuItem.setSelected(true);
          }
          else if (policy == ZoomFitPolicy.WHOLE_PAGE) {
            fitWholePageMenuItem.setSelected(true);
          }

        }
      });
    viewMenu.addSeparator();

    final JCheckBoxMenuItem showThumbnailsMenuItem =
      new JCheckBoxMenuItem(pdfViewerPanel.getPDFShowThumbnailsAction());

    pdfViewerPanel
      .addPDFThumbnailsVisibilityListeners(new PDFThumbnailsVisibilityListener() {

        @Override
        public void onThumbnailsVisibilityChanged(boolean visible) {
          showThumbnailsMenuItem.setSelected(visible);
        }
      });

    viewMenu.add(showThumbnailsMenuItem);

    JMenu helpMenu = new JMenu(getString("help.menu"));
    helpMenu.add(new ApplicationAboutAction(this));

    JMenuBar menuBar = new JMenuBar();
    menuBar.add(fileMenu);
    menuBar.add(viewMenu);
    menuBar.add(helpMenu);

    getApplicationFrame().setJMenuBar(menuBar);
  }

  /**
   * adiciona o listener que escuta mudana sobre a rea de projetos do cliente
   */
  private void addOpenProjFileEnablerListener() {
    ProjectTree projectTree = DesktopFrame.getInstance().getTree();
    projectTree.addProjectTreeListener(openProjFileEnabler);
    openProjFileAction.setEnabled(projectTree.getProject() != null);

  }

  /**
   * remove o listener que escuta mudana sobre a rea de projetos do cliente
   */
  private void removeOpenProfFileEnablerListener() {
    ProjectTree projectTree = DesktopFrame.getInstance().getTree();
    projectTree.removeProjectTreeListener(openProjFileEnabler);
  }

  /**
   * monta a toolbar da aplicao
   */
  private void setupToolBar() {
    JToolBar appToolBar = new JToolBar();
    appToolBar.setFloatable(false);

    appToolBar.add(openProjFileAction);

    if (canOpenLocalFiles()) {
      appToolBar.add(openLocalFileAction);
    }
    appToolBar.addSeparator();

    // retira a toolbar do painel e coloca na aplicao
    appToolBar.add(pdfViewerPanel.getPDFToolBar());
    getApplicationFrame().add(appToolBar, BorderLayout.NORTH);
  }

  /**
   * Retorna true se a aplicao pode abrir arquivos locais, false caso
   * contrrio
   * 
   * @return true se a aplicao pode abrir arquivos locais, false caso
   *         contrrio
   */
  boolean canOpenLocalFiles() {
    return !getApplicationRegistry().requireProject();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
    pdfViewerPanel.closePDF();
    removeOpenProfFileEnablerListener();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Abre um arquivo PDF na aplicao
   * 
   * @param clientFile arquivo PDF a ser aberto
   */
  public void openPdfFile(final ClientFile clientFile) {
    RemoteTask<PDFDocument> loadPdfDocumentTask =
      new RemoteTask<PDFDocument>() {
        @Override
        protected void performTask() throws Exception {
          try {
            InputStream inputStream = clientFile.getInputStream();
            PDFDocument document = new PDFReader().read(inputStream);
            setResult(document);
          }
          catch (final IOException ex) {
            /**
             * InvokeLater  usado para caso este mtodo seja chamado dentro de
             * uma task, no prender a janela da Task ao exibir a mensagem de
             * erro
             */
            SwingThreadDispatcher.invokeLater(new Runnable() {
              @Override
              public void run() {
                StandardDialogs.showErrorDialog(buildApplicationFrame(),
                  getString("pdfviewer.open.error.title"),
                  getString("pdfviewer.open.error") + ex.getMessage());
              }
            });
          }
          finally {
            clientFile.close(true);
          }
        }
      };
    final String taskTitle = getName();
    final String taskDescription = getString("loading.file.message");
    loadPdfDocumentTask.execute(getApplicationFrame(), taskTitle,
      taskDescription);
    if (loadPdfDocumentTask.getStatus()) {
      pdfViewerPanel.loadDocument(loadPdfDocumentTask.getResult());
      setTitleFilename(clientFile.getName());
    }
  }

  /**
   * Fecha o documento corrente
   */
  public void closePdfFile() {
    pdfViewerPanel.closePDF();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(final String name, final Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(ApplicationProject.PROJECT_FILE_MESSAGE)) {
      ClientProjectFile clientFile = (ClientProjectFile) value;
      openPdfFile(clientFile);
    }
  }

  /**
   * Mostra o nome do arquivo na barra de ttulo da aplicao
   * 
   * @param filename nome do arquivo
   */
  private void setTitleFilename(String filename) {
    String windowTitle = getName();
    if (filename != null) {
      windowTitle += " - " + filename;
    }
    getApplicationFrame().setTitle(windowTitle);
  }
}
