package csbase.client.applications.projectsmanager.models;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.logic.User;

/**
 * Classe que representa um bloco de mensagem a ser includa no histrico de um
 * projeto.
 * 
 * @author jnlopes
 * 
 */
public class ProjectHistoryBlock {

  /**
   * Tipo da operao a qual o projeto foi submetido.
   */
  final private ProjectHistoryBlockType blockType;

  /**
   * Lista de usurios que foram manipulados (apenas utilizada quando o projeto
   *  compartilhado).
   */
  private List<User> users = null;

  /**
   * Comentrio includo pelo usurio (apenas utilizado quando uma mensagem 
   * adicionada explicitamente).
   */
  private String userComment;

  /**
   * Construtor
   * 
   * @param blockType Tipo da operao a qual o projeto foi submetido.
   */
  public ProjectHistoryBlock(final ProjectHistoryBlockType blockType) {
    this.blockType = blockType;
  }

  /**
   * Monta o bloco completo da mensagem e a retorna.
   * 
   * @return O bloco completo da mensagem.
   */
  public String getHistoryBlock() {

    final DateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
    final Date date = new Date();

    final User currentUser = User.getLoggedUser();

    final StringBuilder blockHeader =
      new StringBuilder(getString("ProjectHistoryBlock.block.header.text"));
    blockHeader.append(" - " + dateFormat.format(date) + "\n");
    blockHeader.append(getString("ProjectHistoryBlock.user.text") + " "
      + currentUser.getName() + " (" + currentUser.getLogin() + ")\n");

    blockHeader.append(getBlockTypeString());

    if (blockType.equals(ProjectHistoryBlockType.PROJECT_SHARED)) {
      blockHeader.append("\n" + getUsersString());
    }

    blockHeader.append("\n\n");
    return blockHeader.toString();
  }

  /**
   * Caso o projeto seja compartilhado, monta a string que informa os usurios
   * compartilhados.
   * 
   * @return Os usurios compartilhados.
   */
  private String getUsersString() {

    final StringBuilder usersString = new StringBuilder();

    if (users == null) {
      return "";
    }

    for (User u : users) {
      usersString.append(u.getName() + " (" + u.getLogin() + "), ");
    }

    return usersString.substring(0, usersString.length() - 2);
  }

  /**
   * Monta o trecho referente ao tipo da operao a qual o projeto foi
   * submetido.
   * 
   * @return Trecho referente ao tipo da operao a qual o projeto foi
   *         submetido.
   */
  private String getBlockTypeString() {

    final StringBuilder description =
      new StringBuilder(blockType.getDescription());

    if (blockType.equals(ProjectHistoryBlockType.PERSONAL_TEXT)) {
      description.append(" " + userComment);
    }

    return description.toString();
  }

  /**
   * Mtodo de busca de uma string de internacionalizao
   * 
   * @param tag tag desejada
   * @return o texto de internacionalizao
   */
  final private String getString(final String tag) {
    return ProjectsManagerUI.getString(tag);
  }

  /**
   * Define a lista de usurios envolvidos (apenas utilizada quando o projeto
   * for compartilhado).
   * 
   * @param users a lista de usurios
   */
  public void setUsers(final List<User> users) {
    this.users = users;
  }

  /**
   * Define a mensagem adicionada pelo usurio.
   * 
   * @param userComment Mensagem adicionada pelo usurio.
   */
  public void setUserComment(final String userComment) {
    this.userComment = userComment;
  }

}
