package csbase.client.applications.sgamonitor;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationComponentDialog;
import csbase.logic.CapacityType;
import csbase.logic.SGASet;

/**
 * Dilogo que exibe as informaes de um SGA simples.
 * 
 * @author Tecgraf
 */
public class SimpleServerMonitor extends ApplicationComponentDialog<SGAMonitor>
  implements Observer {

  /**
   * Nome do cluster monitorado.
   */
  private String sgaName;

  /**
   * Referncia para a aplicao pai deste dilogo.
   */
  private Application app;

  /**
   * Campo que exibe o nome do SGA.
   */
  private JTextField serverNameField;

  /**
   * Campo que exibe o estado do SGA.
   */
  private JTextField statusField;

  /**
   * Campo que exibe a plataforma em que o SGA est em execuo.
   */
  private JTextField platformField;

  /**
   * Campo que exibe o nmero de processadores da mquina hospedeira do SGA.
   */
  private JTextField nProcessorsField;

  /**
   * Campo que exibe a capacidade da CPU da mquina hospedeira do SGA.
   */
  private JTextField cpuCapacityField;

  /**
   * Campo que exibe a memria RAM (Mb) da mquina hospedeira do SGA.
   */
  private JTextField ramMbField;

  /**
   * Campo que exibe a memria SWAP (Mb) da mquina hospedeira do SGA.
   */
  private JTextField swapMbField;

  /**
   * Campo que exibe o consumo atual da memria RAM (%) da mquina hospedeira.
   */
  private JTextField ramField;

  /**
   * Campo que exibe o consumo atual da memria SWAP (%) da mquina hospedeira.
   */
  private JTextField swapField;

  /**
   * Campo que exibe a carga da CPU no ltimo minuto.
   */
  private JTextField cpuLoad1Field;

  /**
   * Campo que exibe a carga da CPU nos ltimos 5 minutos.
   */
  private JTextField cpuLoad5Field;

  /**
   * Campo que exibe a carga da CPU nos ltimos 15 minutos.
   */
  private JTextField cpuLoad15Field;

  /**
   * Construtor padro.
   * 
   * @param app - aplicao pai.
   */
  public SimpleServerMonitor(SGAMonitor app) {
    super(app);

    this.app = app;
    buildInterface();

    setMinimumSize(new Dimension(500, 500));
    setVisible(false);
    pack();
  }

  /**
   * Atualiza os dados da tabela de ns de um cluster
   * 
   * @param sga - atualiza os dados da tabela de ns de um cluster.
   */
  public void updateSGAInfo(final SGASet sga) {
    sgaName = sga.getName();

    final Application application = getApplication();
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        setTitle(String.format(application.getString("sga.monitor"), sgaName));
        // campos do painel superior
        serverNameField.setText(sga.getName());
        statusField.setText(application.getString("status."
          + sga.getStatus().toString()));
        platformField.setText(sga.getPlatformId());

        // campos do painel central
        nProcessorsField.setText(Integer.toString(sga.getNumProcessors()));

        long cpuCapacity = sga.getCapacity(CapacityType.CPU);
        String cpuCapacityStr = "---";
        if (cpuCapacity >= 0) {
          cpuCapacityStr = Long.toString(cpuCapacity);
        }
        cpuCapacityField.setText(cpuCapacityStr);

        // campos do painel inferior
        ramMbField.setText(formatMemoryAndStorage(sga.getRAMMemoryInfoMb()));
        swapMbField.setText(formatMemoryAndStorage(sga.getSwapMemoryInfoMb()));
        ramField.setText((int) (100.0 - sga.getRAMFreeMemory()) + " %");
        swapField.setText((int) (100.0 - sga.getSwapFreeMemory()) + " %");

        cpuLoad1Field.setText((int) sga.getCPULoad1() + " %");
        cpuLoad5Field.setText((int) sga.getCPULoad5() + " %");
        cpuLoad15Field.setText((int) sga.getCPULoad15() + " %");
      }
    });
  }

  /**
   * Formata um dado inteiro que representa o tamanho de memria em MB, em uma
   * string com o respectivo sufixo (MB, GB ou TB).
   * 
   * @param size - tamanho da memria.
   * @return string com o sufixo adequado.
   */
  private String formatMemoryAndStorage(int size) {
    long s = (long) size * 1024 * 1024;
    return FormatUtils.formatSize(s, 2, getLocale());
  }

  /**
   * Constri a interface que exibe os dados de um SGA simples.
   */
  private void buildInterface() {

    Container container = getContentPane();
    container.setLayout(new GridBagLayout());

    // adicionando painel superior
    container
      .add(createTopPanel(), new GBC(0, 0).insets(10, 10, 10, 10).both());
    container.add(new JSeparator(JSeparator.HORIZONTAL), new GBC(0, 1)
      .right(10).left(10).horizontal());

    // adicionando painel central
    container.add(createCenterPanel(), new GBC(0, 2).insets(10, 10, 10, 10)
      .both());
    container.add(new JSeparator(JSeparator.HORIZONTAL), new GBC(0, 3)
      .right(10).left(10).horizontal());

    // adicionando painel inferior
    container.add(createBottomPanel(), new GBC(0, 4).insets(10, 10, 10, 10)
      .both());

    // Adicionando boto que fecha o dilogo.
    JButton close = new JButton(getApplication().getString("button.close"));
    close.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        SimpleServerMonitor.this.setVisible(false);
      }
    });

    container.add(close, new GBC(0, 5).none().east().insets(0, 0, 10, 20));

  }

  /**
   * Mtodo auxiliar que cria o painel superior do dilogo.
   * 
   * @return painel superior.
   */
  private JPanel createTopPanel() {
    JPanel panel = new JPanel(new GridBagLayout());

    JLabel sgaNameLabel = new JLabel(app.getString("sga.name"));
    serverNameField = new JTextField();
    serverNameField.setEditable(false);
    panel.add(sgaNameLabel, new GBC(0, 0).bottom(5).west().none());
    panel.add(serverNameField, new GBC(1, 0).left(5).bottom(5).horizontal());

    JLabel statusLabel = new JLabel(app.getString("sga.status"));
    statusField = new JTextField();
    statusField.setEditable(false);
    panel.add(statusLabel, new GBC(0, 1).bottom(5).west().none());
    panel.add(statusField, new GBC(1, 1).left(5).bottom(5).horizontal());

    JLabel platformLabel = new JLabel(app.getString("sga.platform"));
    platformField = new JTextField();
    platformField.setEditable(false);
    panel.add(platformLabel, new GBC(0, 2).bottom(5).west().none());
    panel.add(platformField, new GBC(1, 2).left(5).bottom(5).horizontal());

    return panel;
  }

  /**
   * Mtodo auxiliar que cria o painel central do dilogo.
   * 
   * @return painel central.
   */
  private JPanel createCenterPanel() {
    JPanel panel = new JPanel(new GridBagLayout());

    JLabel cpuProcessingLabel = new JLabel(app.getString("cpu.processing"));
    panel.add(cpuProcessingLabel, new GBC(0, 0).bottom(10).west().none());

    JLabel nProcessors = new JLabel(app.getString("number.of.processors"));
    nProcessorsField = new JTextField();
    nProcessorsField.setEditable(false);
    panel.add(nProcessors, new GBC(0, 1).left(20).bottom(5).west().none());
    panel.add(nProcessorsField, new GBC(1, 1).left(5).bottom(5).horizontal());

    JLabel cpuCapacityLabel = new JLabel(app.getString("cpu.capacity"));
    cpuCapacityField = new JTextField();
    cpuCapacityField.setEditable(false);
    panel.add(cpuCapacityLabel, new GBC(0, 2).left(20).bottom(5).west().none());
    panel.add(cpuCapacityField, new GBC(1, 2).left(5).bottom(5).horizontal());

    return panel;
  }

  /**
   * Mtodo auxiliar que cria o painel inferior do dilogo. Esse painel inferior
   *  composto por dois paineis que so dispostos horizontalmente: painel da
   * direita e painel da esquerda.
   * 
   * @return painel inferior.
   */
  private JPanel createBottomPanel() {
    //criando o painel da esquerda
    JPanel left = new JPanel(new GridBagLayout());

    JLabel memoryLabel = new JLabel(app.getString("memory"));
    left.add(memoryLabel, new GBC(0, 0).bottom(10).west().none());

    JLabel memorySizeLabel = new JLabel(app.getString("memory.size"));
    left.add(memorySizeLabel, new GBC(0, 1).left(20).bottom(5).west().none());

    JLabel ramMbLabel = new JLabel(app.getString("memory.ram"));
    ramMbField = new JTextField();
    ramMbField.setEditable(false);
    left.add(ramMbLabel, new GBC(0, 2).left(40).bottom(5).west().none());
    left.add(ramMbField, new GBC(1, 2).left(5).bottom(5).horizontal());

    JLabel swapMbLabel = new JLabel(app.getString("memory.swap"));
    swapMbField = new JTextField();
    swapMbField.setEditable(false);
    left.add(swapMbLabel, new GBC(0, 3).left(40).bottom(5).west().none());
    left.add(swapMbField, new GBC(1, 3).left(5).bottom(5).horizontal());

    JLabel memoryLoadLabel = new JLabel(app.getString("memory.load"));
    left.add(memoryLoadLabel, new GBC(0, 4).left(20).bottom(5).west().none());

    JLabel ramLabel = new JLabel(app.getString("memory.ram"));
    ramField = new JTextField();
    ramField.setEditable(false);
    left.add(ramLabel, new GBC(0, 5).left(40).bottom(5).west().none());
    left.add(ramField, new GBC(1, 5).left(5).bottom(5).horizontal());

    JLabel swapLabel = new JLabel(app.getString("memory.swap"));
    swapField = new JTextField();
    swapField.setEditable(false);
    left.add(swapLabel, new GBC(0, 6).left(40).bottom(5).west().none());
    left.add(swapField, new GBC(1, 6).left(5).bottom(5).horizontal());

    //criando o painel da direita
    JPanel right = new JPanel(new GridBagLayout());

    JLabel cpuLoadLabel = new JLabel(app.getString("cpu.load"));
    right.add(cpuLoadLabel, new GBC(0, 0).bottom(10).west().none());

    JLabel cpuLoad1Label = new JLabel(app.getString("cpu.load.1"));
    cpuLoad1Field = new JTextField();
    cpuLoad1Field.setEditable(false);
    right.add(cpuLoad1Label, new GBC(0, 1).left(20).bottom(5).west().none());
    right.add(cpuLoad1Field, new GBC(1, 1).left(5).bottom(5).horizontal());

    JLabel cpuLoad5Label = new JLabel(app.getString("cpu.load.5"));
    cpuLoad5Field = new JTextField();
    cpuLoad5Field.setEditable(false);
    right.add(cpuLoad5Label, new GBC(0, 2).left(20).bottom(5).west().none());
    right.add(cpuLoad5Field, new GBC(1, 2).left(5).bottom(5).horizontal());

    JLabel cpuLoad15Label = new JLabel(app.getString("cpu.load.15"));
    cpuLoad15Field = new JTextField();
    cpuLoad15Field.setEditable(false);
    right.add(cpuLoad15Label, new GBC(0, 3).left(20).bottom(5).west().none());
    right.add(cpuLoad15Field, new GBC(1, 3).left(5).bottom(5).horizontal());

    // posicionando os paineis da esquerda e direita no painel resultante
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(left, new GBC(0, 0).bottom(10).horizontal());
    panel.add(new JSeparator(JSeparator.VERTICAL),
      new GBC(1, 0).insets(10, 10, 10, 10).vertical());
    panel.add(right, new GBC(2, 0).bottom(10).horizontal());

    return panel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  @SuppressWarnings("unchecked")
  public void update(Observable o, Object arg) {

    List<SGASet> sgaSetList = (List<SGASet>) arg;
    for (SGASet sgaSet : sgaSetList) {
      if (sgaSet.getName().equals(this.sgaName)) {
        updateSGAInfo(sgaSet);
      }
    }
  }

}
