/*
 * $Id: NotificationReplyFrame.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.desktop;

import java.awt.Container;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.util.DateTranslator;
import csbase.logic.User;

/**
 * Tela para exibio de notificao recebida, com painel para envio de
 * resposta.
 * 
 * @author Tecgraf
 */
public class NotificationReplyFrame extends AbstractNotificationFrame {

  /**
   * Indica se deve ou no mostrar o painel de reply
   */
  private boolean showReplyPanel = false;

  /**
   * Label do boto "Cancelar".
   */
  private static final String CANCEL_BUTTON_LABEL = LNG
    .get("notification.cancel.button");
  /**
   * Label do boto "Fechar".
   */
  private static final String CLOSE_BUTTON_LABEL = LNG
    .get("notification.close.button");
  /**
   * Login do remetente.
   */
  private final String senderLogin;
  /**
   * Contedo da mensagem recebida.
   */
  private final String receivedMessage;
  /**
   * Data e hora do envio da mensagem.
   */
  private final Date timestamp;
  /**
   * Identificadores dos destinatrios (sempre conter apenas o remetente).
   */
  private final List<Object> recipientsIDs;
  /**
   * Flag que indica se houve erro ao recuperar o ID do remetente.
   */
  private boolean senderError;

  /**
   * Construtor.
   * 
   * @param senderLogin login do remetente
   * @param date data de envio da mensagem
   * @param receivedMessage contedo da mensagem
   */
  public NotificationReplyFrame(final String senderLogin, final Date date,
    final String receivedMessage) {
    this.senderLogin = senderLogin;
    this.timestamp = date;
    this.receivedMessage = receivedMessage;
    recipientsIDs = new ArrayList<Object>();
    senderError = false;
    User sender;
    try {
      sender = User.getUserByLogin(senderLogin);
      if (sender != null) {
        recipientsIDs.add(sender.getId());
        showReplyPanel = true;
      }
    }
    catch (Exception e) {
      StandardDialogs.showErrorDialog(DesktopFrame.getInstance()
        .getDesktopFrame(), "notification.reply.error.title",
        "notification.reply.error.badSender");
      senderError = true;
    }

    buildUI();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<Object> getRecipientsIDs() {
    return recipientsIDs;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ActionListener getSendButtonActionListener() {
    return new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        sendMessage(true);
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean hasRecipients() {
    return !senderError;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void addSubPanels(Container mainPane) {
    /*
     * painel com a mensagem recebida
     */
    JPanel receivedMsgPane = createReceivedMessagePane();
    GBC gbc = new GBC().both().northwest().insets(10);
    mainPane.add(receivedMsgPane, gbc);
    /*
     * painel para responder  mensagem. S  mostrado quando o sender  vlido.
     */
    if (showReplyPanel) {
      JPanel replyMsgPane =
        createEditionPanel(LNG.get("notification.reply.panel.title"));
      gbc = new GBC(0, 1).both().northwest().insets(0, 10, 10, 10);
      mainPane.add(replyMsgPane, gbc);
    }
    /*
     * painel com os botes
     */
    JPanel buttonsPane = createButtonsPane(showReplyPanel);
    gbc = new GBC(0, 2).center().bottom(10);
    mainPane.add(buttonsPane, gbc);
  }

  /**
   * Cria o painel com os dados da mensagem recebida.
   * 
   * @return painel com os dados da mensagem recebida
   */
  private JPanel createReceivedMessagePane() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.setBorder(BorderFactory.createTitledBorder(LNG
      .get("notification.compose.panel.title")));
    GBC gbc;
    final JLabel senderLabel =
      new JLabel(LNG.get("notification.sender.label") + ':');
    gbc = new GBC(0, 0).west().insets(5, 10, 0, 5);
    panel.add(senderLabel, gbc);

    final JLabel senderName = new JLabel(senderLogin);
    Font boldFont = senderName.getFont();
    Font plainFont = boldFont.deriveFont(Font.PLAIN);
    senderName.setFont(plainFont);
    gbc = new GBC(1, 0).west().insets(5, 0, 0, 30);
    panel.add(senderName, gbc);

    final JLabel dateLabel =
      new JLabel(LNG.get("notification.date.label") + ':');
    gbc = new GBC(2, 0).west().insets(5, 0, 0, 5);
    panel.add(dateLabel, gbc);

    final JLabel dateText =
      new JLabel((DateTranslator.getInstance()).translate(timestamp));
    dateText.setFont(plainFont);
    gbc = new GBC(3, 0).west().top(5);
    panel.add(dateText, gbc);

    // TextArea "Mensagem"
    final JTextArea msgTextArea = new JTextArea();
    msgTextArea.setText(receivedMessage);
    msgTextArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 12));
    msgTextArea.setEditable(false);
    msgTextArea.setRows(10);
    msgTextArea.setColumns(80);
    msgTextArea.setCaretPosition(0);
    JScrollPane msgScrollPane = new JScrollPane(msgTextArea);
    gbc = new GBC(0, 1).both().width(4).northwest().insets(10);
    panel.add(msgScrollPane, gbc);

    return panel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void preShowInit() {
    closeButton.setText(CLOSE_BUTTON_LABEL);
    popUpToggle.setSelected(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getWindowTitle() {
    return LNG.get("notification.reply.title");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void messageTextChanged() {
    if (editionArea.getText().isEmpty()) {
      closeButton.setText(CLOSE_BUTTON_LABEL);
    }
    else {
      closeButton.setText(CANCEL_BUTTON_LABEL);
    }
  }
}
