package csbase.client.project;

import java.awt.Window;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.kernel.ClientException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileFilter;
import csbase.logic.filters.ProjectFileCompositeAndFilter;
import csbase.logic.filters.ProjectFileDirectoryFilter;
import csbase.logic.filters.ProjectFileNameFilter;
import csbase.logic.filters.ProjectFileNotFilter;
import csbase.logic.filters.ProjectFileTrueFilter;
import csbase.logic.filters.ProjectFileTypeFilter;

/**
 * Essa classe implementa um navegador para abertura de um ou mais tipos de
 * arquivos de um projeto. Pode-se configurar se o navegador vai permitir, por
 * exemplo, a seleo mltipla de arquivos, ou se vai permitir somente a seleo
 * de arquivos e no de diretrios.
 */
public class ProjectFileChooserOpen extends ProjectFileChooser {

  /**
   * Contri um navegador para abertura de um ou mais arquivos de um projeto, de
   * acordo com o modo de seleo especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode) throws ClientException {
    this(owner, project, isMultiSelectionEnabled, mode, null);
  }

  /**
   * Contri um navegador para abertura de um ou mais arquivos de um projeto,
   * que sejam de um determinado tipo, de acordo com o modo de seleo
   * especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @param fileType tipo de arquivo default do navegador
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode, String fileType)
    throws ClientException {
    this(owner, project, isMultiSelectionEnabled, mode, fileType, false);
  }

  /**
   * Contri um navegador para abertura de um ou mais arquivos de um projeto,
   * que sejam de um determinado tipo, de acordo com o modo de seleo
   * especificado para o navegador.
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @param fileType tipo de arquivo default do navegador
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode, String fileType,
    boolean useFilter) throws ClientException {
    this(owner, project, isMultiSelectionEnabled, mode, new String[]{fileType},
      fileType, useFilter);
  }

  /**
   * Contri um navegador para abertura de um ou mais tipos de arquivos de um
   * projeto. O modo de seleo no navegador permite somente a seleo de
   * arquivos (@see ProjectFileChooser.FILE_ONLY).
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param fileTypes array contendo os tipos de arquivos permitidos
   * @param defaultType tipo default que vai aparecer selecionado na lista
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, String[] fileTypes, String defaultType)
    throws ClientException {
    this(owner, project, isMultiSelectionEnabled, ProjectFileChooser.FILE_ONLY,
      fileTypes, defaultType);
  }

  /**
   * Contri um navegador para abertura de um ou mais tipos de arquivos de um
   * projeto. O modo de seleo no navegador permite somente a seleo de
   * arquivos (@see ProjectFileChooser.FILE_ONLY).
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @param fileTypes array contendo os tipos de arquivos permitidos
   * @param defaultType tipo default que vai aparecer selecionado na lista
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
    boolean isMultiSelectionEnabled, int mode, String[] fileTypes,
    String defaultType)
    throws ClientException {
    this(owner, project, isMultiSelectionEnabled, mode, fileTypes, defaultType, false);
  }

  /**
   * Contri um navegador para abertura de um ou mais tipos de arquivos de um
   * projeto. O modo de seleo no navegador permite somente a seleo de
   * arquivos (@see ProjectFileChooser.FILE_ONLY).
   * 
   * @param owner janela pai
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled se for true, indica que permite seleo
   *        mltipla de arquivos para abertura, caso contrrio, somente um
   *        arquivo pode ser selecionado
   * @param mode modo de seleo no navegador (@see ProjectFileChooser)
   * @param fileTypes array contendo os tipos de arquivos permitidos
   * @param defaultType tipo default que vai aparecer selecionado na lista
   * @param useFilter se um panel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  public ProjectFileChooserOpen(Window owner, CommonClientProject project,
      boolean isMultiSelectionEnabled, int mode, String[] fileTypes,
      String defaultType, boolean useFilter)
    throws ClientException {
    super(owner, LNG.get("PRJ_FILE_CHOOSER_OPEN_TITLE"), project,
        isMultiSelectionEnabled, mode, defaultType, createVisualFilter(mode,
          false, null, fileTypes),
          createSelectionFilter(mode, null, fileTypes), false, null,
        useFilter);
    String title = getTitle();
    if (title != null) {
      dialog.setTitle(title);
    }
    dialog.setVisible(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter() {
    return getSelectionFilter(null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter(String fileType) {
    return getSelectionFilter(null, fileType);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected ProjectFileFilter getSelectionFilter(final String nameFilter,
    final String fileType) {
    return createSelectionFilter(mode, nameFilter, new String[]{fileType});
  }

  private static ProjectFileFilter createSelectionFilter(int mode, String name,
    String[] fileTypes) {
    boolean useNameFilter = name != null && !name.isEmpty();
    boolean useTypeFilter = fileTypes != null && fileTypes.length != 0;

    ProjectFileCompositeAndFilter andFilter = new ProjectFileCompositeAndFilter();

    if (useNameFilter) {
      andFilter.addChild(new ProjectFileNameFilter(name));
    }

    if (useTypeFilter) {
      andFilter.addChild(new ProjectFileTypeFilter(fileTypes));
    }

    switch (mode) {
      case FILE_ONLY:
        // Seleciona somente arquivos.
        andFilter.addChild(new ProjectFileNotFilter(ProjectFileDirectoryFilter.getInstance()));
        break;
      case DIRECTORY_ONLY:
        // Seleciona somente diretrios.
        andFilter.addChild(ProjectFileDirectoryFilter.getInstance());
        break;
      case FILE_AND_DIRECTORY:
        // Seleciona todos.
        andFilter.addChild(ProjectFileTrueFilter.getInstance());
        break;
      default:
        String msg = String.format("Modo invlido %s.", mode);
        throw new IllegalArgumentException(msg);
    }

    return andFilter;
  }
  
  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean handleAction() {
    ClientProjectFile[] selectedFiles = this.getSelectedFiles();
    if (selectedFiles != null) {
      this.selectedPaths = new ProjectTreePath[selectedFiles.length];
      for (int i = 0; i < this.selectedPaths.length; i++) {
        this.selectedPaths[i] = new ProjectTreePath(selectedFiles[i]);
      }
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void makeSelection(ClientProjectFile[] projectFileArray) {
    if (projectFileArray.length == 0) {
      this.selectedPaths = null;
      this.fileNameText.setText("");
    }
    else {
      this.selectedPaths = new ProjectTreePath[projectFileArray.length];
      String newText = "";
      String separator = "";
      for (int i = 0; i < projectFileArray.length; i++) {
        this.selectedPaths[i] = new ProjectTreePath(projectFileArray[i]);
        newText += (separator + projectFileArray[i].getName());
        separator = ",";
      }
      this.fileNameText.setText(newText);
    }
  }
}
