/*
 * ClientUtilities.java
 * 
 * $Author: clinio $ $Revision: 173963 $ - $Date: 2007-03-22 18:23:16 -0300
 * (qui, 22 mar 2007) $
 */
package csbase.client.util;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoManager;

import csbase.client.Client;
import csbase.client.applications.ApplicationImages;
import tecgraf.javautils.core.lng.LNG;

/**
 * A classe <code>ClientUtilities</code> prov uma coleo de mtodos teis para
 * as interfaces do sistema.
 * 
 * @version $Revision: 173963 $
 */
public class ClientUtilities {
  /**
   * Adiciona o nome do sistema a um ttulo de janela, usando o formato: "titulo
   * - nome do sistema". Provisoriamente, o nome do sistema  determinado pela
   * propriedade "SERVER", obtida no arquivo de idiomas. O nome dessa
   * propriedade  inadequado, porm ela  usada intensamente no cdigo, e
   * redefinida pelos sistemas construdos a partir do CSBase. Quando possvel,
   *  conveniente alterar o nome dessa propriedade (ou criar uma outra
   * propriedade para isso).
   * 
   * @param title o ttulo da janela
   * 
   * @return o ttulo acrescido do nome do sistema
   */
  public static String addSystemNameToTitle(String title) {
    return title + " - " + LNG.get("SERVER");
  }

  /**
   * Adiciona o nome do sistema a um ttulo de janela, usando o formato: "titulo
   * - nome do sistema". Provisoriamente, o nome do sistema  determinado pela
   * propriedade "SERVER", obtida no arquivo de idiomas. O nome dessa
   * propriedade  inadequado, porm ela  usada intensamente no cdigo, e
   * redefinida pelos sistemas construdos a partir do CSBase. Quando possvel,
   *  conveniente alterar o nome dessa propriedade (ou criar uma outra
   * propriedade para isso).
   * 
   * @param title o ttulo da janela
   * 
   * @return o ttulo acrescido do nome do sistema
   */
  public static String addSystemAndServerNameToTitle(String title) {
    return title + " - " + LNG.get("SERVER") + " (" + Client.getInstance()
      .getServerName() + ")";
  }

  /**
   * Ajuste de um conjunto de elementos de interface para o mesmo tamanho.
   * 
   * @param comps os widgets que sero ajustados.
   */
  public static void adjustEqualSizes(JComponent... comps) {
    final Dimension dim = new Dimension(0, 0);
    for (int i = 0; i < comps.length; i++) {
      final Dimension pref = comps[i].getPreferredSize();
      final double h = Math.max(dim.getHeight(), pref.getHeight());
      final double w = Math.max(dim.getWidth(), pref.getWidth());
      dim.setSize(w, h);
    }
    for (int i = 0; i < comps.length; i++) {
      comps[i].setPreferredSize(dim);
    }
  }

  /**
   * Mtodo para a criao de botes baseados em imagens, j devidamente
   * ajustados em seus tamanhos etc.
   * 
   * @param icon Objeto <code>ImageIcon</code> usado como cone.
   * 
   * @return .
   */
  public static JButton createImageButton(ImageIcon icon) {
    JButton button = new JButton();
    button.setIcon(icon);
    trimImageButton(button);
    return button;
  }

  /**
   * Mtodo que ajusta o tamanho do boto para que ele seja do tamanho razovel
   * (praticamente o mnimo) para exibir o seu cone.
   * 
   * @param button O boto.
   */
  public static void trimImageButton(JButton button) {
    Icon icon = button.getIcon();
    Dimension size = new Dimension(icon.getIconWidth() + 2, icon.getIconHeight()
      + 2);
    button.setPreferredSize(size);
    button.setMaximumSize(size);
    button.setMinimumSize(size);
  }

  /**
   * Aviso de erro de preenchimento de campo qualquer com abertura de pop-up,
   * seleo do texto indicado e alerta no label correspondente.
   * 
   * @param window .
   * @param msg mensagem de erro
   * @param button JButton a ser selecionado para nova edio do usurio.
   */
  public static void showErrorByButton(Window window, String msg,
    JButton button) {
    button.setIcon(ApplicationImages.ICON_STOP_16);
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(window, msg, addSystemNameToTitle(LNG.get(
      "UTILITIES_INPUT_ERROR")), JOptionPane.ERROR_MESSAGE);
    button.requestFocus();
    button.setIcon(null);
  }

  /**
   * Aviso de erro de preenchimento de campo qualquer com abertura de pop-up,
   * seleo do texto indicado e alerta no label correspondente.
   * 
   * @param window .
   * @param msg mensagem de erro
   * @param label label a ser marcado com flag de erro.
   * @param component JComponent a ser selecionado para nova edio do usurio.
   */
  public static void showErrorByComponent(Window window, String msg,
    JLabel label, JComponent component) {
    if (label != null) {
      label.setIcon(ApplicationImages.ICON_STOP_16);
    }
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(window, msg, addSystemNameToTitle(LNG.get(
      "UTILITIES_INPUT_ERROR")), JOptionPane.ERROR_MESSAGE);
    if (component != null) {
      component.requestFocus();
    }
    if (label != null) {
      label.setIcon(null);
    }
  }

  /**
   * Aviso de erro de preenchimento de campo qualquer com abertura de pop-up,
   * seleo do texto indicado e alerta no label correspondente.
   * 
   * @param window .
   * @param exception Exceo
   * @param label label a ser marcado com flag de erro.
   * @param component JComponent a ser selecionado para nova edio do usurio.
   */
  public static void showErrorByComponent(Window window, Exception exception,
    JLabel label, JComponent component) {
    if (label != null) {
      label.setIcon(ApplicationImages.ICON_STOP_16);
    }
    Toolkit.getDefaultToolkit().beep();
    StandardErrorDialogs.showErrorDialog(window, exception);
    if (component != null) {
      component.requestFocus();
    }
    if (label != null) {
      label.setIcon(null);
    }
  }

  /**
   * Adiciona um componente grfico do Swing em um panel usando o GridBagLayout
   * que ser configurado pelos parmetros.
   * 
   * @param panel referncia para o painel que conter o componente
   * @param component componente a ser adicionado de acordo com restries
   * @param c referncia para o objeto que representa as restries do
   *        GridBagLayout
   * @param gx posio horizontal do elemento no grid
   * @param gy posio vertical do elemento no grid
   * @param w numero de clulas em uma linha (span horizontal)
   * @param h numero de clulas em uma coluna (span vertical)
   * @param wx weightx
   * @param wy weighty
   */
  public static void addComponent2Panel(JPanel panel, JComponent component,
    GridBagConstraints c, int gx, int gy, int w, int h, int wx, int wy) {
    c.gridx = gx;
    c.gridy = gy;
    c.gridwidth = w;
    c.gridheight = h;
    c.weightx = wx;
    c.weighty = wy;
    panel.add(component, c);
  }

  /**
   * Aviso de erro de preenchimento de campo de text-box com abertura de pop-up,
   * seleo do texto indicado e alerta no label correspondente.
   * 
   * @param window .
   * @param msg mensagem de erro
   * @param label label a ser marcado com flag de erro.
   * @param field text a ser selecionado para nova edio do usurio.
   */
  public static void showErrorByTextField(Window window, String msg,
    JLabel label, JTextField field) {
    if (label != null) {
      label.setIcon(ApplicationImages.ICON_STOP_16);
    }
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(window, msg, addSystemNameToTitle(LNG.get(
      "UTILITIES_INPUT_ERROR")), JOptionPane.ERROR_MESSAGE);
    if (field != null) {
      field.requestFocus();
      field.setSelectionStart(0);
      field.setSelectionEnd(field.getText().length());
    }
    if (label != null) {
      label.setIcon(null);
    }
  }

  /**
   * Aviso de erro de preenchimento de clula de tabela com abertura de pop-up,
   * seleo do texto indicado e alerta no label correspondente.
   * 
   * @param window .
   * @param msg mensagem de erro
   * @param label label a ser marcado com flag de erro.
   * @param table tabela a ser selecionada para nova edio do usurio.
   * @param row linha a ser selecionada para nova edio do usurio.
   * @param col coluna a ser selecionada para nova edio do usurio.
   */
  public static void showErrorByTable(Window window, String msg, JLabel label,
    JTable table, int row, int col) {
    if (label != null) {
      label.setIcon(ApplicationImages.ICON_STOP_16);
    }
    Toolkit.getDefaultToolkit().beep();
    JOptionPane.showMessageDialog(window, msg, addSystemNameToTitle(LNG.get(
      "UTILITIES_INPUT_ERROR")), JOptionPane.ERROR_MESSAGE);
    if (table != null) {
      table.requestFocus();
      table.clearSelection();
      table.changeSelection(row, col, true, true);
    }
    if (label != null) {
      label.setIcon(null);
    }
  }

  /**
   * Verifica se o path  composto apenas por caracteres alfanumricos,
   * separador de diretorio (/), underscore (_) ou ponto (.) ou traco (-).
   * 
   * @param path Nome a ser validado
   * 
   * @return true se o nome estiver de acordo com as especificaes do sistema
   */
  public static boolean validatePathChars(String path) {
    if (path.equals("") || !path.matches("^[\\w\\.\\-/]+$")) {
      return false;
    }
    return true;
  }

  /**
   * Mtodo para carga de strings de texto baseado em um Locale. OBS.: Esse
   * mtodo foi movido para essa classe para ser possvel disponibizar os
   * recursos de mensagens para aplicaes que no usam o LoginFrame do csbase
   * 
   * @param locale Locale selecionado pelo usurio.
   * 
   * @return flag indicativo de sucesso da operao.
   */
  public static boolean loadStringMap(Locale locale) {
    String[] propertiesFiles = {
        "csbase.client.resources.properties.language.metal",
        "csbase.client.resources.properties.language.basic",
        "csbase.client.resources.properties.language.awt" };
    ResourceBundle rb;
    for (int i = 0; i < propertiesFiles.length; i++) {
      try {
        rb = ResourceBundle.getBundle(propertiesFiles[i], locale);
        for (Enumeration e = rb.getKeys(); e.hasMoreElements();) {
          String key = (String) e.nextElement();
          UIManager.put(key, rb.getString(key));
        }
      }
      catch (Exception e) {
        return false;
      }
    }
    return true;
  }

  /**
   * Mtodo para carga de strings de mensagens baseado em um Locale. OBS.: Esse
   * mtodo foi movido para essa classe para ser possvel disponibizar os
   * rescursos de mensagens para aplicaes que no usam o LoginFrame do csbase
   * 
   * @param locale Locale selecionado pelo usurio.
   */
  public static void setLanguage(Locale locale) {
    String idiomFile = "csbase.client.resources.properties.language.idiom";
    LNG.load(idiomFile, locale);
  }

  /**
   * Verifica se o nome  composto apenas por caracteres alfanumricos,
   * underscore (_) ou ponto (.) ou traco (-).
   * 
   * @param name Nome a ser validado
   * 
   * @return true se o nome estiver de acordo com as especificaes do sistema
   */
  public static boolean isValidFileName(String name) {
    if (name.equals("") || !name.matches("^[\\w\\.\\-]+$")) {
      return false;
    }
    return true;
  }

  /**
   * Centraliza uma janela em relao a outra janela de referncia. Se a janela
   * de referncia no estiver sendo exibida, a referncia  substituda pela
   * janela que a contm (seu owner) at que seja encontrada uma janela que
   * esteja sendo exibida, ou que se chegue ao fim da sequncia. Se a janela de
   * referncia for nula, ou no houver janela de referncia sendo exibida,
   * centraliza em relao  tela.
   * 
   * @param window janela a ser centralizada.
   * @param refWindow janela de referncia.
   */
  public static void centerWindow(Window window, Window refWindow) {
    while (refWindow != null && !refWindow.isVisible()) {
      refWindow = refWindow.getOwner();
    }
    Dimension size = null;
    Point local = null;
    if (refWindow != null) {
      size = refWindow.getSize();
      local = refWindow.getLocationOnScreen();
    }
    else {
      size = Toolkit.getDefaultToolkit().getScreenSize();
      local = new Point(0, 0);
    }
    int x = (size.width / 2) + local.x;
    int y = (size.height / 2) + local.y;
    Dimension wsize = window.getSize();
    x -= wsize.width / 2;
    y -= wsize.height / 2;
    window.setLocation(x, y);
  }

  /**
   * Recupera um comparador de strings que ignora letras com acento, maisculas
   * e minsculas. Esse comparador implementa Serializable.
   * 
   * @return
   */
  public static final Comparator getStringComparatorIgnoreCase() {
    return new StringComparatorIgnoreCase();
  }

  /**
   * Aplica e gerencia aes de 'desfazer' e 'refazer' no componente de texto
   * indicado.
   * <ul>
   * <li>CTRL + Z = desfazer
   * <li>CTRL + Y = refazer
   * </ul>
   * 
   * @param textcomp - o componente de texto que ser incrementado.
   */
  public static void applyUndoRedoActions(JTextComponent textcomp) {
    final UndoManager undoManager = new UndoManager();
    Document doc = textcomp.getDocument();

    // ouvinte de eventos de 'undo' e 'redo'
    doc.addUndoableEditListener(new UndoableEditListener() {
      @Override
      public void undoableEditHappened(UndoableEditEvent evt) {
        undoManager.addEdit(evt.getEdit());
      }
    });

    /////////////
    // DESFAZER:
    textcomp.getActionMap().put("Undo", new AbstractAction("Undo") {
      @Override
      public void actionPerformed(ActionEvent evt) {
        try {
          if (undoManager.canUndo()) {
            undoManager.undo();
          }
        }
        catch (CannotUndoException e) {
        }
      }
    });
    textcomp.getInputMap().put(KeyStroke.getKeyStroke("control Z"), "Undo");

    /////////////
    // REFAZER:
    textcomp.getActionMap().put("Redo", new AbstractAction("Redo") {
      @Override
      public void actionPerformed(ActionEvent evt) {
        try {
          if (undoManager.canRedo()) {
            undoManager.redo();
          }
        }
        catch (CannotRedoException e) {
        }
      }
    });
    textcomp.getInputMap().put(KeyStroke.getKeyStroke("control Y"), "Redo");
  }

  /**
   * Classe interna que representa um comparador de Strings, ignorando letras
   * com acento, maisculas e minsculas.
   * 
   * @author ntl9
   */
  private static class StringComparatorIgnoreCase implements Comparator<String>,
    Serializable {
    /**
     * {@inheritDoc}
     */
    @Override
    public int compare(String s1, String s2) {
      final String n1 = removeSpecialChar(s1);
      final String n2 = removeSpecialChar(s2);
      return n1.compareToIgnoreCase(n2);
    }

    /**
     * Retira de string caracteres especiais
     * 
     * @param str string
     * @return novo texto
     */
    private static String removeSpecialChar(String str) {
      if (str != null && str.trim().length() > 0) {
        String[] replaces = { "a", "e", "i", "o", "u", "c" };
        Pattern[] patternsArray = new Pattern[replaces.length];
        patternsArray[0] = Pattern.compile("[]", Pattern.CASE_INSENSITIVE);
        patternsArray[1] = Pattern.compile("[]", Pattern.CASE_INSENSITIVE);
        patternsArray[2] = Pattern.compile("[]", Pattern.CASE_INSENSITIVE);
        patternsArray[3] = Pattern.compile("[]", Pattern.CASE_INSENSITIVE);
        patternsArray[4] = Pattern.compile("[]", Pattern.CASE_INSENSITIVE);
        patternsArray[5] = Pattern.compile("[]", Pattern.CASE_INSENSITIVE);
        for (int i = 0; i < patternsArray.length; i++) {
          Matcher matcher = patternsArray[i].matcher(str);
          str = matcher.replaceAll(replaces[i]);
        }
      }
      return str;
    }
  }
}
