package csbase.client.util.gui;

import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;

import javax.swing.JPanel;

/**
 * A classe <code>CardPanel</code>  um painel que simplifica o uso do
 * {@link CardLayout}, alm de adicionar funcionalidades ao uso deste. Todos os
 * mtodos de CardLayout para navegao entre "cartas" (componentes) tem mtodos
 * equivalentes nesta classe.<br>
 * Funcionalidades adicionadas:
 * <ul>
 * <li>Possibilidade de tamanho horizontal e/ou vertical dinmico, ou seja,
 * redimensionamento horizontal e/ou vertical do painel a cada troca de "carta".
 * </li>
 * <li>Manipulao dos componentes sem precisar atribuir nome a estes; basta
 * passar a referncia para o componente.</li>
 * <li>Recuperao do componente que est em exibio</li>
 */
public class CardPanel extends JPanel {

  /** LayoutManager deste painel. */
  private final CardLayout cardLayout;

  /**
   * Classe que sincroniza a lista de referncias para os componentes e a lista
   * de nomes que referenciam os componentes.
   */
  private CardsList cardsList;

  /**
   * Valor boleano que indica se o tamanho vertical deste painel  dinmico, ou
   * seja, se  reajustado a cada troca de "carta".
   */
  boolean dynamicVSize;

  /**
   * Valor boleano que indica se o tamanho horizontal deste painel  dinmico,
   * ou seja, se  reajustado a cada troca de "carta".
   */
  boolean dynamicHSize;

  /**
   * Cria um CardPanel com dimenses estticas.
   */
  public CardPanel() {
    this(false, false);
  }

  /**
   * Cria um CardPanel que pode ter tamanho horizontal e/ou vertical dinmico,
   * de acordo com o valor dos parmetros deste construtor.
   * 
   * @param dynamicVSize Valor boleano que indica se o tamanho vertical deste
   *        painel  dinmico, ou seja, se  reajustado a cada troca de "carta".
   * @param dynamicHSize Valor boleano que indica se o tamanho horizontal deste
   *        painel  dinmico, ou seja, se  reajustado a cada troca de "carta".
   */
  public CardPanel(boolean dynamicVSize, boolean dynamicHSize) {
    this.dynamicVSize = dynamicVSize;
    this.dynamicHSize = dynamicHSize;
    cardLayout = initializeCardLayout();
    super.setLayout(cardLayout);
    cardsList = new CardsList();
  }

  /**
   * @return Instncia de CardLayout com o mtodo preferredLayoutSize(...)
   *         sobrescrito para implementar o comportamento de tamanho horizontal
   *         e/ou vertical dinmico.
   */
  private CardLayout initializeCardLayout() {
    return new CardLayout() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Dimension preferredLayoutSize(Container parent) {

        if (!dynamicVSize && !dynamicHSize) {
          return super.preferredLayoutSize(parent);
        }

        Dimension d = new Dimension();
        if (!dynamicVSize || !dynamicHSize) {
          d = super.preferredLayoutSize(parent);
        }

        Component comp = CardPanel.this.getCurrentCard();
        if (comp == null) {
          return new Dimension(0, 0);
        }
        int h;
        int w;

        h = dynamicVSize ? comp.getPreferredSize().height : d.height;
        w = dynamicHSize ? comp.getPreferredSize().width : d.width;

        Insets insets = parent.getInsets();

        return new Dimension(insets.left + insets.right + w, insets.top
          + insets.bottom + h);
      }
    };
  }

  /**
   * @return O componente correspondente  "carta" que est em exibio, ou
   *         <code>null</code>, caso a lista de "cartas" esteja vazia.
   */
  public Component getCurrentCard() {
    return cardsList.getCurrentCard();
  }

  /**
   * @return O nome correspondente  "carta" que est em exibio, ou
   *         <code>null</code>, caso a lista de "cartas" esteja vazia.
   */
  public String getCurrentCardName() {
    Component comp = cardsList.getCurrentCard();
    return cardsList.getCardName(comp);
  }

  /**
   * Muda o componente em exibio para o componente cujo nome corresponde ao
   * nome recebido como parmetro, caso o componente a ser exibido tenha sido
   * previamente adicionado  este painel. Se o nome recebido no corresponder a
   * um componente adicionado  este painel, nada acontecer, e o mtodo
   * retornar <code>null</code>. <br>
   * Obs.: A utilizao deste mtodo  recomendada apenas se, ao adicionar o
   * componente  este painel, um nome foi associado  este.
   * 
   * @param name Nome associado ao componente a ser exibido.
   * @return Referncia para o componente exibido, caso o nome recebido
   *         corresponda a um componente adicionado  este painel, ou
   *         <code>null</code>, caso contrrio.
   * @see CardPanel#add(Component, Object)
   * @see CardPanel#show(String)
   */
  public Component show(String name) {
    Component comp = cardsList.updateCurrentCard(name);
    if (comp != null) {
      cardLayout.show(this, name);
      return getCurrentCard();
    }
    return null;
  }

  /**
   * Muda o componente em exibio para o componente cuja referncia foi
   * recebida como parmetro, caso o componente a ser exibido tenha sido
   * previamente adicionado  este painel. Se a referncia recebida no
   * corresponder a um componente adicionado  este painel, nada acontecer, e o
   * mtodo retornar <code>false</code>.
   * 
   * @param comp Referncia ao componente a ser exibido.
   * @return True, caso a referncia recebida corresponda a um componente
   *         adicionado  este painel. False, caso contrrio.
   * @see CardPanel#add(Component)
   * @see CardPanel#show(String)
   */
  public boolean show(Component comp) {
    String name = cardsList.getCardName(comp);
    if (name != null) {
      cardLayout.show(this, name);
      cardsList.updateCurrentCard(name);
      return true;
    }
    return false;
  }

  /**
   * Adiciona o componente  lista de componentes (cartas) deste painel,
   * associando-o ao nome recebido como parmetro. Caso j exista referncia ao
   * componente ou nome recebido na lista de "cartas" deste painel, nada
   * acontecer.
   * 
   * @param comp Referncia ao componente a ser adicionado  este painel.
   * @param name Nome associado ao componente.
   * @see CardPanel#add(Component)
   */
  @Override
  public void add(Component comp, Object name) {
    this.add(comp, name, -1);
  }

  /**
   * Adiciona o componente  lista de componentes (cartas) deste painel,
   * associando-o a um nome criado arbitrariamente pelo mtodo
   * {@link CardsList#add(Component)}. Caso j exista referncia ao componente
   * recebido na lista de "cartas" deste painel, nada acontecer, e o mtodo
   * retornar <code>null</code>.
   * 
   * @param comp Referncia ao componente a ser adicionado  este painel.
   * @return O componente inserido, caso o componente tenha sido adicionado com
   *         sucesso, ou <code>null</code>, caso contrrio.
   * @see CardPanel#add(Component, Object)
   */
  @Override
  public Component add(Component comp) {
    return this.add(comp, -1);
  }

  /**
   * Adiciona o componente  lista de componentes (cartas) deste painel,
   * associando-o a um nome criado arbitrariamente pelo mtodo
   * {@link CardsList#add(Component)}. Caso j exista referncia ao componente
   * recebido na lista de "cartas" deste painel, nada acontecer, e o mtodo
   * retornar <code>null</code>.
   * 
   * @param comp Referncia ao componente a ser adicionado  este painel.
   * @param index ndice da posio na lista de componentes onde este componente
   *        ser inserido. <code>-1</code> significa que o componente ser
   *        inserido ao final da lista.
   * @return O componente inserido, caso o componente tenha sido adicionado com
   *         sucesso, ou <code>null</code>, caso contrrio.
   * @see CardPanel#add(Component, Object)
   */
  @Override
  public Component add(Component comp, int index) {
    String name = cardsList.add(comp, index);
    if (name != null) {
      super.add(comp, name, index);
      return comp;
    }
    return null;
  }

  /**
   * Adiciona o componente  lista de componentes (cartas) deste painel,
   * associando-o ao nome recebido como parmetro. Caso j exista referncia ao
   * componente ou nome recebido na lista de "cartas" deste painel, nada
   * acontecer.
   * 
   * @param comp Referncia ao componente a ser adicionado  este painel.
   * @param name Nome associado ao componente.
   * @param index ndice da posio na lista de componentes onde este componente
   *        ser inserido. <code>-1</code> significa que o componente ser
   *        inserido ao final da lista.
   * @see CardPanel#add(Component)
   */
  @Override
  public void add(Component comp, Object name, int index) {
    if (name instanceof String) {
      if (cardsList.add((String) name, comp, index)) {
        super.add(comp, name, index);
        return;
      }
    }
    this.add(comp, index);
  }

  /**
   * Remove o componente correspondente  este nome da lista de "cartas" deste
   * painel. Caso a carta removida seja a carta corrente, a carta corrente
   * passar a ser a prxima.
   * 
   * @param name Nome do componente a ser removido.
   */
  public void remove(String name) {
    Component comp = cardsList.getComponentForCardName(name);
    if (comp != null) {
      remove(comp);
    }
  }

  /**
   * Remove o componente correspondente  referncia recebida como parmetro da
   * lista de "cartas" deste painel. Caso a carta removida seja a carta
   * corrente, a carta corrente passar a ser a prxima.
   * 
   * @param comp Referncia ao componente a ser removido.
   */
  @Override
  public void remove(Component comp) {
    super.remove(comp);
    cardsList.remove(comp);
  }

  /**
   * Exibe o componente seguinte da lista de "cartas". Se o componente corrente
   * for o ltimo da lista, o componente corrente passa a ser o primeiro
   * componente da lista.
   * 
   * @return O componente exibido, ou <code>null</code>, caso a lista de
   *         "cartas" esteja vazia.
   */
  public Component next() {
    Component comp = cardsList.getNextCard();
    show(comp);
    return comp;
  }

  /**
   * Exibe o componente anterior da lista de "cartas". Se o componente corrente
   * for o primeiro da lista, o componente corrente passa a ser o ultimo
   * componente da lista.
   * 
   * @return O componente exibido, ou <code>null</code>, caso a lista de
   *         "cartas" esteja vazia.
   */
  public Component previous() {
    Component comp = cardsList.getPreviousCard();
    show(comp);
    return comp;
  }

  /**
   * Exibe o primeiro componente da lista de "cartas".
   * 
   * @return O componente exibido, ou <code>null</code>, caso a lista de
   *         "cartas" esteja vazia.
   */
  public Component first() {
    Component comp = cardsList.getFirstCard();
    show(comp);
    return comp;
  }

  /**
   * Exibe o ltimo componente da lista de "cartas".
   * 
   * @return O componente exibido, ou <code>null</code>, caso a lista de
   *         "cartas" esteja vazia.
   */
  public Component last() {
    Component comp = cardsList.getLastCard();
    show(comp);
    return comp;
  }

  /**
   * Verifica se existe um componente associado ao nome.
   * 
   * @param name - nome do componente.
   * @return true se existir um componente associado ao nome, false caso
   *         contrrio.
   */
  public boolean has(String name) {
    return cardsList.getComponentForCardName(name) != null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setLayout(LayoutManager mgr) {
    // vazio -- usamos sempre CardLayout
  }

}
