package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.ReplicateSharedDialog;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Ao para cpia de esquema de compartilhamento entre projetos.
 * 
 * @author Tecgraf
 */
public class ReplicateSharingDataAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager a aplicao
   */
  public ReplicateSharingDataAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   * 
   * @throws Exception em caso de erro.
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() == 0) {
      return;
    }

    final ApplicationFrame owner = projectsManager.getApplicationFrame();
    /*
     * Se o usurio selecionou mltiplos projetos, avisamos antes que as
     * alteraes sobrescrevero as configuraes de a todos os projetos
     * selecionados
     * 
     * configurao de permisses de acesso s pode ser feita para um projeto
     * por vez
     */
    if (projectList.size() > 1) {
      StandardDialogs.showInfoDialog(owner,
        LNG.get("OpenSharingDialogAction.info.dialog.title"),
        LNG.get("OpenSharingDialogAction.info.dialog.message"));
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    /*
     *  garantido que s existe um projeto selecionado
     */
    final ProjectsManagerData sourceProject = projects.get(0);
    final ReplicateSharedDialog rsd =
      new ReplicateSharedDialog(projectsManager, sourceProject);
    rsd.pack();
    //    rsd.center();
    rsd.setVisible(true);

    final Set<ProjectsManagerData> targetProjects = rsd.getSelectedProjects();
    if (targetProjects == null || targetProjects.size() == 0) {
      return;
    }

    final ReplicateSharingDataTask rst =
      new ReplicateSharingDataTask(projectsManager, sourceProject,
        targetProjects);

    final String title = getString("ReplicateSharingDataAction.title");
    final String msg = getString("ReplicateSharingDataAction.message");
    rst.execute(projectsManager.getApplicationFrame(), title, msg);

    if (rst.wasCancelled()) {
      final String err =
        getString("ReplicateSharingDataAction.cancelled.message");
      rst.showError(err);
      return;
    }

    if (rst.getStatus() != true) {
      final Exception exception = rst.getError();
      throw exception;
    }

    final SharingType sharingType = sourceProject.getSharingType();
    final Set<Object> uRO = sourceProject.getUsersRO();
    final Set<Object> uRW = sourceProject.getUsersRW();
    for (ProjectsManagerData pmd : targetProjects) {
      pmd.setSharingType(sharingType);
      pmd.setUsers(uRO, uRW);
    }
    projectsManager.refreshProjectsTable();
    projectsManager.refreshInfoPanel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    /*
     * FIXME - Substituir cone!
     */
    //  return ProjectsManagerUI.DEFINE_USERS;
    return ApplicationImages.ICON_COPY_16;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    final ProjectsManagerData pmd = projectList.get(0);

    final ProjectSpaceAllocation spaceAllocation =
      pmd.getProjectSpaceAllocation();

    final boolean isAdmin = loggedUser.isAdmin();
    final boolean isOwner = pmd.getOwnerId().equals(loggedUserId);

    /*
     * Para que um usurio possa alterar o compartilhamento de um projeto 
     * preciso que: - O projeto possa ser aberto (no esteja aguardando alocao
     * / desalocao). - O usurio seja o administrador ou o dono do projeto.
     */
    if (spaceAllocation.isOpenable() && (isAdmin || isOwner)) {
      final List<ProjectsManagerData> l = new ArrayList<ProjectsManagerData>();
      l.add(pmd);
      return l;
    }

    final String deniedMessage =
      getString("ReplicateSharingDataAction.project.selection.denied.message")
        + getString("ReplicateSharingDataAction.project.requirements.message");

    StatusDialog
      .showNoneOkDialog(
        getProjectsManager(),
        deniedMessage,
        (List<ProjectsManagerData>) null,
        getString("ReplicateSharingDataAction.project.selection.failure.message"));

    return null;
  }
}

/**
 * Task que copia configuraes de compartilhamento de um projeto a outros.
 * 
 * @author Tecgraf
 */
class ReplicateSharingDataTask extends ProjectsManagerTask<Void> {

  /**
   * Projeto cuja configurao de compartilhamento ser copiada a outros
   * projetos.
   */
  final private ProjectsManagerData sourceProject;

  /**
   * Projetos que tero suas configuraes de compartilhamento sobreescritas.
   */
  final private Set<ProjectsManagerData> targetProjects;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao.
   * @param sourceProject Projeto origem.
   * @param targetProjects Projetos destino.
   */
  public ReplicateSharingDataTask(final ProjectsManager projectsManager,
    final ProjectsManagerData sourceProject,
    final Set<ProjectsManagerData> targetProjects) {
    super(projectsManager);
    this.sourceProject = sourceProject;
    this.targetProjects = targetProjects;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    CommonClientProject sourceCCP =
      psi.openProject(sourceProject.getProjectId(), false);
    Set<Object> usersRO = sourceCCP.getUsersRO();
    Set<Object> usersRW = sourceCCP.getUsersRW();
    SharingType sharingType = sourceCCP.getSharingType();
    sourceCCP.close(false);
    for (ProjectsManagerData pmd : targetProjects) {
      CommonClientProject targetCCP = psi.openProject(pmd.getProjectId(), true);
      targetCCP.updateUsers(sharingType, usersRO, usersRW);
      targetCCP.close(true);
    }
  }
}
