package csbase.client.rest.resources.v1;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.Vector;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import com.fasterxml.jackson.databind.ObjectMapper;

import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.rest.ExtendedResponseStatus;
import csbase.logic.applicationservice.ApplicationRegistry;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.annotations.Authorization;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe que implementa o endpoint server para aplicaes Mtodos nesse servio
 * so exportados a partir da URL base applications;
 *
 * Created by tiago on 10/3/16.
 */

@Path("v1/applications")
@Api(value = "Applications")
public class ApplicationRestService {

  /**
   * Mapper de objetos.
   */
  private static final ObjectMapper objectMapper = new ObjectMapper();

  /**
   * Classe que representa uma aplicao que pode ser lanada via API Rest do
   * cliente.
   *
   * Esse tipo de recurso possui dos campos, um nome legvel e um tipo. O tipo
   * deve ser usado para referenciar aplicaes;
   */
  public static final class ApplicationResource {
    /**
     * Nome
     */
    private String name;

    /**
     * Tipo
     */
    private String type;

    /**
     * Consulta o nome.
     *
     * @return nome
     */
    public String getName() {
      return name;
    }

    /**
     * Consulta o tipo.
     *
     * @return o tipo
     */
    public String getType() {
      return type;
    }

    /**
     * Construtor
     *
     * @param name nome
     * @param type tipo
     */
    public ApplicationResource(final String name, final String type) {
      this.name = name;
      this.type = type;
    }
  }

  /**
   * Classe que representa aplicaes em execuo no cliente.
   *
   * Esse tipo de recurso possui trs campos, o nome legvel da aplicao, o seu
   * tipo e uma lista de identificadores de aplicaes desse tipo em execuo.
   */
  @ApiModel(value = "test")
  public static final class RunningApplicationResource {
    /**
     * Nome
     */
    private String name;

    /**
     * Tipo
     */
    private String type;

    /**
     * Lista de instncias
     */
    private List<String> instanceId;

    /**
     * Consulta o nome
     *
     * @return nome
     */
    @ApiModelProperty(value = "The name of the application", required = true)
    public String getName() {
      return name;
    }

    /**
     * Consulta a lista de instncia
     *
     * @return lista
     */
    @ApiModelProperty(value = "The list of application instances Ids", required = true)
    public List<String> getInstanceId() {
      return instanceId;
    }

    /**
     * Ajuste da lista de instncias.
     *
     * @param instanceId
     */
    public void setInstanceId(List<String> instanceId) {
      this.instanceId = instanceId;
    }

    /**
     * Consulta o tipo.
     *
     * @return o tipo
     */
    @ApiModelProperty(value = "The application type", required = true)
    public String getType() {
      return type;
    }

    /**
     * Construtor
     *
     * @param name nome
     * @param type tipo
     */
    public RunningApplicationResource(final String name, final String type) {
      this.name = name;
      this.type = type;
    }
  }

  /**
   * Classe que representa respostas Rest para mensagens sncronas enviadas a
   * apliaes;
   *
   * Essa classe possui dois campos: o identificador da instncia que est
   * respondendo a requisio e a resposta da chamada propriamente dita.
   *
   */
  public static final class MessageResponse {
    /**
     * Id da instncia
     */
    private final String instanceId;

    /**
     * Response
     */
    private final Object response;

    /**
     * Construtor
     *
     * @param instanceId id da instncia
     * @param response response
     */
    public MessageResponse(String instanceId, Object response) {
      this.instanceId = instanceId;
      this.response = response;
    }

    /**
     * Consuiltao id da instncia
     *
     * @return id
     */
    public String getInstanceId() {
      return instanceId;
    }

    /**
     * Consulta respponse.
     *
     * @return resposnse
     */
    public Object getResponse() {
      return response;
    }
  }

  /**
   * Classe que representa respostas de erro Rest para mensagens;
   */
  public static final class RESTApplicationInstance {
    /**
     * Tipo da aplicao
     */
    private final String type;

    /**
     * ID da instncia da aplicao
     */
    private final String instanceId;

    /**
     * Construtor
     *
     * @param type tipo
     * @param instanceId id de instncia
     */
    public RESTApplicationInstance(String type, String instanceId) {
      this.type = type;
      this.instanceId = instanceId;
    }

    /**
     * Retorna o tipo da aplicao.
     *
     * @return o tipo
     */
    @ApiModelProperty(value = "The application type", required = true)
    public String getType() {
      return type;
    }

    /**
     * Retorna o id da instncia.
     *
     * @return o id
     */
    @ApiModelProperty(value = "The application instance ID", required = true)
    public String getInstanceId() {
      return instanceId;
    }

  }

  /**
   * Classe que representa respostas de erro Rest para mensagens;
   */
  public static final class RESTError {
    /**
     * Texto do erro
     */
    private final String error;

    /**
     * Construtor
     *
     * @param error
     */
    public RESTError(String error) {
      this.error = error;
    }

    /**
     * Retorna o texto.
     *
     * @return o texto
     */
    @ApiModelProperty(value = "The returned error description", required = true)
    public String getError() {
      return error;
    }
  }

  /**
   * Endpoint do mtodo GET da chamada Rest /aplications
   *
   * @return Uma lista de recursos de aplicaes disponveis no cliente.
   */
  @GET
  @ApiOperation(value = "Get a list of available applications", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @Produces(MediaType.APPLICATION_JSON)
  public List<ApplicationResource> getApps() {
    List<ApplicationResource> app = new ArrayList<>();

    ApplicationManager am = ApplicationManager.getInstance();
    for (ApplicationRegistry r : am.getAllApplicationRegistries()) {
      final Client client = Client.getInstance();
      final String name = r.getApplicationName(client.getDefaultLocale());
      final String type = r.getId();
      ApplicationResource res = new ApplicationResource(name, type);
      app.add(res);
    }

    return app;
  }

  /**
   * Endpoint do mtodo GET da chamada Rest /aplications/running
   *
   * @return Uma lista de recursos de aplicaes em execuo no cliente.
   */
  @GET
  @Path("/running")
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Get a list of running applications", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  public List<RunningApplicationResource> getRunningApps() {
    List<RunningApplicationResource> app = new ArrayList<>();

    ApplicationManager am = ApplicationManager.getInstance();
    //get set of application types
    final Set<Entry<String, Vector<ApplicationType>>> runningApps = am
      .getRunningApplications().entrySet();

    //iterate vector
    for (Map.Entry<String, Vector<ApplicationType>> r : runningApps) {
      final String type = r.getKey();

      final Locale locale = Client.getInstance().getDefaultLocale();

      //get app registry
      final ApplicationRegistry registry = am.getApplicationRegistry(type);

      final String name = registry.getApplicationName(locale);

      RunningApplicationResource res = new RunningApplicationResource(name,
        type);

      List<String> ids = new ArrayList<>();

      for (ApplicationType t : r.getValue()) {
        ids.add(t.getInstanceId());
      }

      res.setInstanceId(ids);

      if (ids.size() > 0) {
        app.add(res);
      }

    }

    return app;
  }

  /**
   *
   * Endpoint do mtodo DELETE da chamada Rest /aplications
   *
   * Mata uma aplicao em execuo no cliente.
   *
   * @param instanceId O identificador de instncia da aplicao
   * @param force indicativo de desligado forado.
   * @return A resposta REST relativa  requisio
   */

  @DELETE
  @Path("{instanceId}")
  @ApiOperation(value = "Kill a running application", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = RESTApplicationInstance.class),
      @ApiResponse(code = 404, message = "Instance Id is not running", response = RESTError.class),
      @ApiResponse(code = 500, message = "The application threw an exception while shutting down.", response = RESTError.class), })
  public Response killRunningApp(
    @ApiParam(value = "Application instance Id", required = true) @PathParam("instanceId") String instanceId,
    @ApiParam(value = "Force kill application", required = false, example = "true") @QueryParam("force") String force) {
    try {
      //get the app manager
      final ApplicationManager manager = ApplicationManager.getInstance();

      //get app instance
      ApplicationType appInstance = manager.getApplicationInstance(instanceId);

      if (appInstance == null) { //instance not found
        return Response.status(Status.NOT_FOUND).entity(new RESTError(LNG.get(
          "application.manager.instance.not.found"))).build();
      }

      //return structure
      final RESTApplicationInstance data = new RESTApplicationInstance(
        appInstance.getName(), instanceId);

      //if force kill
      if (force != null && force.equals("true")) {
        manager.killApplication(appInstance);
      }
      else {
        //ask nicely
        if(!appInstance.closeApplication()) {
          return Response.status(Status.PRECONDITION_FAILED).entity(data).build();
        }
      }
      return Response.ok().entity(data).build();

    }
    catch (Exception e) {
      e.printStackTrace();
      final String errMsg = e.getLocalizedMessage();
      return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(
        new RESTError(errMsg)).build();
    }
  }

  /**
   * Endpoint do mtodo POST da chamada Rest /aplications
   *
   * Executa uma aplicao no cliente
   *
   * @param appType Tipo da aplicao a ser executada
   *
   * @return A resposta Rest relativa  requisio cliente contendo o
   *         identificador da instncia da aplicao executada.
   */
  @POST
  @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Run an application on the client", notes = "This can only be done by the logged in user.", authorizations = {
      @Authorization(value = "oauth2") })
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = RESTApplicationInstance.class),
      @ApiResponse(code = 404, message = "The application type  was not found", response = RESTError.class),
      @ApiResponse(code = 500, message = "The application threw an exception while initializing.", response = RESTError.class), })
  public Response runApp(
    @ApiParam(value = "Type of the application", required = true) @FormParam("type") String appType) {
    try {
      //Get app manager
      final ApplicationManager manager = ApplicationManager.getInstance();

      //get app registry for the type
      final ApplicationRegistry registry = manager.getApplicationRegistry(
        appType);
      if (registry == null) { //apptype is not valid
        return Response.status(Status.NOT_FOUND).entity(new RESTError(
          "The application type '" + appType + "' not found")).build();
      }

      //Run application by its type
      ApplicationType applicationType = manager.runApplication(appType);

      //return structure
      final RESTApplicationInstance data = new RESTApplicationInstance(appType,
        applicationType.getInstanceId());

      //build response
      return Response.status(Response.Status.OK).entity(data).build();
    }
    catch (ApplicationException ex) {
      return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(
        new RESTError(ex.getLocalizedMessage())).build();
    }
  }

  /**
   * Endpoint do mtodo POST da chamada Rest /aplications/{instanceId}/message
   *
   * Envia uma mensagem sncrona  aplicao. Esse mtodo aguarda a resposta da
   * aplicao.
   *
   * @param instanceId O identificador da instncia que ir tratar a mensagem
   * @param body O dado que dever ser tratado pela aplicao, deve ser um dado
   *        com formato JSON
   *
   * @return A resposta Rest (200 - OK) relativa  mensagem retornada pela
   *         instncia da aplicao. Contm o identificador da instncia da
   *         aplicao e sua resposta em formato JSON, 404 caso a instncia no
   *         seja vlida, 500 caso a aplicao lance uma exceo, ou 400 se a
   *         mensagem no estiver no formato JSON.
   */

  @POST
  @Path("{instanceId}/syncmessage")
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Send a sync message to a running application", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation", response = MessageResponse.class),
      @ApiResponse(code = 400, message = "Required input parameters cannot be validated", response = RESTError.class),
      @ApiResponse(code = 404, message = "Instance Id is not running", response = RESTError.class),
      @ApiResponse(code = 412, message = "The application is on invalid state for processing this type of message", response = RESTError.class),
      @ApiResponse(code = 422, message = "The application rejected the input parameters", response = RESTError.class),
      @ApiResponse(code = 500, message = "The application threw a runtime exception while processing the message.", response = RESTError.class), })
  public Response sendMessageSync(
    @ApiParam(value = "Instance Id of the application", required = true) @PathParam("instanceId") String instanceId,
    @ApiParam(value = "Message body", required = true) String body) {
    final ApplicationManager manager = ApplicationManager.getInstance();
    ApplicationType appInstance = manager.getApplicationInstance(instanceId);
    if (appInstance == null) {
      return Response.status(Status.NOT_FOUND).entity(new RESTError(LNG.get(
        "application.manager.instance.not.found"))).build();
    }
    try {
      HashMap<String, Object> map;
      try {
        map = objectMapper.readValue(body, HashMap.class);
        if (map == null) {
          throw new Exception("empty body");
        }
      }
      catch (Throwable ex) {
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError(LNG.get("csbase.client.rest.server.notjason") + ": "
            + ex.getLocalizedMessage())).build();
      }

      if (!map.containsKey("type") && !(map.get("type") instanceof String)) {
        final String errMsg = LNG.get("csbase.client.rest.server.notype");
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError(errMsg)).build();
      }

      if (!map.containsKey("senderId") && !(map.get(
        "senderId") instanceof String)) {
        final String errMsg = LNG.get("csbase.client.rest.server.nosender");
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError(errMsg)).build();
      }

      Object response = manager.sendSyncMessage(instanceId, (String) map.get(
        "type"), map.get("value"), (String) map.get("senderId"));
      MessageResponse resp = new MessageResponse(instanceId, response);
      return Response.ok().entity(resp).build();

    }
    catch (IllegalArgumentException ex) {
      return Response.status(ExtendedResponseStatus.UNPROCESSABLE_ENTITY)
        .entity(new RESTError(ex.getLocalizedMessage())).build();

    }
    catch (IllegalStateException ex) {
      return Response.status(Response.Status.PRECONDITION_FAILED).entity(
        new RESTError(ex.getLocalizedMessage())).build();
    }
    catch (Throwable ex) {
      return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(
        new RESTError(ex.getLocalizedMessage())).build();
    }
  }

  /**
   * Endpoint do mtodo POST da chamada Rest
   * /aplications/{instanceId}/asyncmessage
   *
   * Envia uma mensagem assncrona  aplicao. Esse mtodo no aguarda a
   * resposta da aplicao.
   *
   * @param instanceId O identificador da instncia que ir tratar a mensagem
   * @param body O dado que dever ser tratado pela aplicao
   * @return Status 200 (OK) caso a mensagem tenha sido enviada com sucesso, ou
   *         404 caso a instncia no seja vlida.
   */
  @POST
  @Path("{appid}/asyncmessage")
  @Produces(MediaType.APPLICATION_JSON)
  @ApiOperation(value = "Send an async message to a running application", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation"),
      @ApiResponse(code = 400, message = "Required input parameters cannot be validated", response = RESTError.class),
      @ApiResponse(code = 404, message = "Instance Id is not running", response = RESTError.class),
      @ApiResponse(code = 412, message = "The application is on invaid state for processing this type of message", response = RESTError.class),
      @ApiResponse(code = 422, message = "The application rejected the input parameters", response = RESTError.class),
      @ApiResponse(code = 500, message = "The application threw a runtime exception while processing the message.", response = RESTError.class), })
  public Response sendMessageAsync(
    @ApiParam(value = "Instance Id of the application", required = true) @PathParam("appid") String instanceId,
    @ApiParam(value = "Message body", required = true) String body) {
    ApplicationType applicationInstance = ApplicationManager.getInstance()
      .getApplicationInstance(instanceId);
    if (applicationInstance == null) {
      final RESTError restErr = new RESTError(LNG.get(
        "application.manager.instance.not.found"));
      return Response.status(Response.Status.NOT_FOUND).entity(restErr).build();
    }
    try {
      HashMap<String, Object> map;

      try {
        map = objectMapper.readValue(body, HashMap.class);
        if (map == null) {
          throw new Exception("empty body");
        }
      }
      catch (Throwable ex) {
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError(LNG.get("csbase.client.rest.server.notjason") + ": "
            + ex.getLocalizedMessage())).build();
      }

      if (!map.containsKey("type") && !(map.get("type") instanceof String)) {
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError(LNG.get("csbase.client.rest.server.notype"))).build();
      }

      if (!map.containsKey("senderId") && !(map.get(
        "senderId") instanceof String)) {
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError(LNG.get("csbase.client.rest.server.nosender"))).build();
      }

      if (instanceId.equals("broadcast")) {
        ApplicationManager.getInstance().broadcastAsyncMessage((String) map.get(
          "type"), map.get("value"), (String) map.get("senderId"));
      }
      else {
        ApplicationManager.getInstance().sendAsyncMessage(instanceId,
          (String) map.get("type"), map.get("value"), (String) map.get(
            "senderId"));
      }
      return Response.ok().entity(new MessageResponse(instanceId, "")).build();

    }
    catch (IllegalArgumentException ex) {
      return Response.status(ExtendedResponseStatus.UNPROCESSABLE_ENTITY)
        .entity(new RESTError(ex.getLocalizedMessage())).build();

    }
    catch (IllegalStateException ex) {
      return Response.status(Response.Status.PRECONDITION_FAILED).entity(
        new RESTError(ex.getLocalizedMessage())).build();
    }
    catch (Throwable ex) {
      return Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(
        new RESTError(ex.getLocalizedMessage())).build();
    }

  }

  /**
   * Endpoint do mtodo POST da chamada Rest /aplications/register
   *
   * Registra uma aplicao externa com um id que pode ser usado no resposta da
   * aplicao.
   *
   * @param appType O identificador da instncia que ir tratar a mensagem
   * @param uri O dado que dever ser tratado pela aplicao
   * @return Status 200 (OK) caso a mensagem tenha sido enviada com sucesso, ou
   *         404 caso a instncia no seja vlida.
   */
  @POST
  @Path("register")
  @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
  @ApiOperation(value = "Register an external application ", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation"),
      @ApiResponse(code = 500, message = "The application threw a runtime exception while registering service.", response = RESTError.class), })
  public Response register(
    @ApiParam(value = "Type of the application to be registered", required = true) @FormParam("type") String appType,
    @ApiParam(value = "The callback URL for handling messages", required = true) @FormParam("uri") String uri) {

    //    System.out.println(body);
    try {
      if (uri == null) {
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError("Callback URI is null")).build();
      }

      if (appType == null) {
        return Response.status(Response.Status.BAD_REQUEST).entity(
          new RESTError("Application type is null")).build();
      }

      final ApplicationManager manager = ApplicationManager.getInstance();
      String appId = manager.registerExternalApplication(uri, appType);
      return Response.ok().entity(appId).build();
    }
    catch (Throwable ex) {
      final RESTError restError = new RESTError(ex.getLocalizedMessage());
      return Response.status(Status.INTERNAL_SERVER_ERROR).entity(restError)
        .build();
    }
  }

  /**
   * Endpoint do mtodo DELETE da chamada Rest /aplications/{appType}/unregister
   *
   * Registra uma aplicao externa com um id que pode ser usado no resposta da
   * aplicao.
   *
   * @param appId O identificador da instncia que ir tratar a mensagem
   * @return Status 200 (OK) caso a mensagem tenha sido enviada com sucesso, ou
   *         404 caso a instncia no seja vlida.
   */
  @DELETE
  @Path("{appId}/unregister")
  @Produces(MediaType.TEXT_PLAIN)
  @ApiOperation(value = "Unregister an external application ", notes = "This can only be done by the logged in user.")
  @ApiImplicitParams({
      @ApiImplicitParam(name = "Authorization", value = "Token (Bearer $TOKEN)", dataType = "string", required = true, paramType = "header") })
  @ApiResponses(value = {
      @ApiResponse(code = 200, message = "Successful operation"),
      @ApiResponse(code = 500, message = "The application threw a runtime exception while registering service.", response = RESTError.class), })
  public Response unregister(
    @ApiParam(value = "The application instance ID to be unregistered", required = true) @PathParam("appId") String appId) {

    try {
      final ApplicationManager manager = ApplicationManager.getInstance();
      manager.unregisterExternalApplication(appId);
      return Response.ok().entity(appId).build();
    }
    catch (Throwable ex) {
      final RESTError restError = new RESTError(ex.getLocalizedMessage());
      return Response.status(Status.INTERNAL_SERVER_ERROR).entity(restError)
        .build();
    }
  }

}
