/*
 * Detalhes da ltima alterao:
 * 
 * $Author: clinio $ $Date: 2008-09-17 10:05:53 -0300 (Wed, 17 Sep 2008) $
 * $Revision: 167902 $
 */
package csbase.client.login;

import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.util.Locale;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.KeyStroke;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.Client;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.exception.CSBaseException;

/**
 * Representa uma interface abstrata de login.  usado pelo cliente para
 * verificar se um usurio pode ou no acessar o sistema.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstractLoginUI {

  /**
   * Indica se o login foi cancelado
   */
  private boolean canceled;

  /**
   * Define o lock utilizado para o fechamento do dilogo.
   */
  final private Lock continueLock = new ReentrantLock();

  /**
   * Define a condio para finalizao do dilogo.
   */
  final private Condition continueCondition = continueLock.newCondition();

  /**
   * Cancela o login. Utilizar o <CODE>setCanceled</CODE> para definir que foi
   * cancelado
   */
  protected abstract void cancel();

  /**
   * Indica se o login foi cancelado
   * 
   * @return indicativo.
   */
  protected boolean isCanceled() {
    return canceled;
  }

  /**
   * Ajuste de indicaitvo de login cancelado.
   * 
   * @param canceled indicativo
   */
  protected void setCanceled(boolean canceled) {
    this.canceled = canceled;
  }

  /**
   * Efetua o login do usurio no servidor.
   * 
   * @param login O login do usurio.
   * @param password A senha do usurio.
   * @param locale O locale escolhido pelo usurio.
   * 
   * @return true, caso o login tenha sido realizado com sucesso, ou false, caso
   *         contrrio.
   * 
   * @throws CSBaseException Caso ocorra algum erro durante o procedimento.
   */
  private boolean doLogin(final String login, final String password,
    final Locale locale) throws CSBaseException {
    ClientRemoteMonitor.getInstance().start(login, password, locale);
    return ClientRemoteMonitor.getInstance().isAlive();
  }

  /**
   * <p>
   * Executa o dilogo.
   * </p>
   * 
   * <p>
   * OBS: No pode ser chamado na thread do Swing.
   * </p>
   * 
   * @param defaultLocale Um {@code Locale} para ser utilizado como default na
   *        combo de idiomas. Se nenhum for especificado, ser utilizado o valor
   *        retornado por {@link Locale#getDefault()}.
   * 
   * @return Os dados informados pelo usurio.
   * 
   * @throws IllegalStateException Caso a execuo do dilogo seja solicitada na
   *         thread do Swing.
   */
  public final InitialContext execute(final Locale defaultLocale) {
    if (SwingThreadDispatcher.isEventDispatchThread()) {
      throw new IllegalStateException(
        "O dilogo de login no pode ser executado na Thread do Swing.");
    }

    final Locale locale =
      defaultLocale == null ? Locale.getDefault() : defaultLocale;
    initializeUI(locale);

    this.continueLock.lock();
    try {
      showUI();
      ClientRemoteMonitor.getInstance().getServerLookupThread().start();
      this.continueCondition.awaitUninterruptibly();
    }
    finally {
      this.continueLock.unlock();
    }
    return this.getLoginData();
  }

  /**
   * Inicializa a interface
   * 
   * @param defaultLocale Locale default para inicializar a interface
   */
  protected abstract void initializeUI(Locale defaultLocale);

  /**
   * Exibe o dilogo de login
   */
  protected abstract void showUI();

  /**
   * Fecha o dilogo de login
   */
  public abstract void disposeUI();

  /**
   * Notifica que ocorreu um erro na carga do cliente
   */
  public void preClientInitializationException() {
  }

  /**
   * Obtm os dados informados pelo usurio.
   * 
   * @return Os dados informados pelo usurio, ou null, caso o usurio tenha
   *         cancelado o dilogo.
   */
  public abstract InitialContext getLoginData();

  /**
   * Ao executada no momento em que o login  iniciado.
   */
  protected abstract void updateUIForLoginStarting();

  /**
   * Obtm o login do usurio
   * 
   * @return String
   */
  protected abstract String getLogin();

  /**
   * Obtm a senha do usurio
   * 
   * @return String
   */
  protected abstract String getPassword();

  /**
   * Obtm o locale escolhido para o idioma
   * 
   * @return Locale
   */
  protected abstract Locale getSelectedLocale();

  /**
   * Valida/atualiza a interface. Executado no momento em que o login 
   * soliciatado
   * 
   * @return <CODE>TRUE</CODE> para permitir que o login continue sendo feito,
   *         <CODE>FALSE</CODE> para abortar e continuar na tela de login
   */
  protected abstract boolean validateLoginUI();

  /**
   * Atualiza a interface aps o login ter sido iniciado e terminado sem
   * sucesso.
   */
  protected abstract void updateUIForInvalidLoginInfo();

  /**
   * Atualiza a interface aps o login ter sido iniciado e terminado por causa
   * de alguma incompatibilidade de verso de cliente
   */
  protected abstract void updateUIForInvalidClientVersion();

  /**
   * Atualiza a interface aps o login ter sido iniciado e terminado por causa
   * de algum erro
   */
  protected abstract void updateUIForLoginException();

  /**
   * Efetua o login.
   */
  protected final void login() {
    if (validateLoginUI() == false) {
      return;
    }

    final String login = getLogin();
    final String password = getPassword();

    updateUIForLoginStarting();

    final Thread t = new Thread() {
      @Override
      public void run() {
        try {
          if (!doLogin(login, password, getSelectedLocale())) {
            updateUIForInvalidLoginInfo();
          }
          else {
            String clientVersion = Client.getInstance().getVersion();
            String serverVersion = ClientRemoteMonitor.getInstance().getServer()
              .getVersionName();
            if (!clientVersion.equals(serverVersion)) {
              updateUIForInvalidClientVersion();
              System.err.println("Verso cliente: " + clientVersion);
              System.err.println("Verso servidor: " + serverVersion);
            }
            else {
              setCanceled(false);
              unlock();
            }
          }
        }
        catch (final Exception e) {
          e.printStackTrace();
          updateUIForLoginException();
        }
      }
    };

    t.start();
  }

  /**
   * Ao a ser executada aps a tela de login (cliente comea a ser
   * inicializado)
   */
  public void preClientInitialization() {
  }

  /**
   * Libera o lock que mantm o dilogo aberto.
   */
  protected final void unlock() {
    this.continueLock.lock();
    try {
      this.continueCondition.signalAll();
    }
    finally {
      this.continueLock.unlock();
    }
  }

  /**
   * Seta o login do usurio
   * 
   * @param login o login
   */
  protected abstract void setLogin(String login);

  /**
   * Seta a senha do usurio
   * 
   * @param password a senha
   */
  protected abstract void setPassword(String password);

  /**
   * Define aes default para as teclas.
   * 
   * <UL>
   * <LI>ENTER: efetua o login</LI>
   * <LI>ESC: cancela o login</LI>
   * <LI>CTRL+SHIFT+ENTER: seta "admin" e "1234" como login/password e efeuta o
   * login</LI>
   * </UL>
   * 
   * @param frame janela do login
   */
  protected void registerDefaultActions(JFrame frame) {
    // Ao para ENTER (OK)
    KeyStroke okKeyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
    addStrokeAction(frame, okKeyStroke, new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        login();
      }
    });

    // Ao para ESC (cancelamento)
    KeyStroke cancelKeyStroke = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
    addStrokeAction(frame, cancelKeyStroke, new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        cancel();
      }
    });

    // "admin speed login": CTRL+SHIFT+ENTER = admin + 1234 + ENTER
    KeyStroke adminLoginCombo =
      KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.CTRL_DOWN_MASK
        | InputEvent.SHIFT_DOWN_MASK);
    addEasterEggSpeedLogin(frame, adminLoginCombo, "admin");

    // "user speed login": CTRL+ALT+ENTER = $USER + 1234 + ENTER
    KeyStroke userLoginCombo =
      KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.CTRL_DOWN_MASK
        | InputEvent.ALT_DOWN_MASK);
    addEasterEggSpeedLogin(frame, userLoginCombo, System.getenv("USER"));
  }

  /**
   * Adiciona um easter egg para speed login.
   * 
   * @param frame frane
   * @param keyStroke stroke (teclado)
   * @param login login usado.
   */
  private void addEasterEggSpeedLogin(JFrame frame, KeyStroke keyStroke,
    String login) {
    JComponent component = frame.getRootPane();
    int condition = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    InputMap inputMap = component.getInputMap(condition);
    ActionMap actionMap = component.getActionMap();

    final String loginText = login;
    AbstractAction userLoginAction = new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (loginText == null || loginText.trim().isEmpty()) {
          return;
        }
        setLogin(loginText);
        setPassword("1234");
        login();
      }
    };
    String actionMapKey = keyStroke.toString();
    inputMap.put(keyStroke, actionMapKey);
    actionMap.put(actionMapKey, userLoginAction);
  }

  /**
   * Adiciona um easter egg para speed login.
   * 
   * @param frame frane
   * @param keyStroke stroke (teclado)
   * @param action ao.
   */
  protected void addStrokeAction(JFrame frame, KeyStroke keyStroke,
    AbstractAction action) {
    JComponent component = frame.getRootPane();
    int condition = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    InputMap inputMap = component.getInputMap(condition);
    ActionMap actionMap = component.getActionMap();

    String actionMapKey = keyStroke.toString();
    inputMap.put(keyStroke, actionMapKey);
    actionMap.put(actionMapKey, action);
  }
}
