package csbase.client.applicationmanager;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.rmi.RemoteException;

import csbase.client.Client;
import csbase.client.desktop.DesktopPref;
import csbase.client.desktop.RemoteTask;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.types.PVBoolean;
import csbase.client.preferences.types.PVString;
import csbase.logic.FileInfo;
import csbase.logic.SyncRemoteFileChannel;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.remote.ApplicationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;

/**
 * Controla um diretrio de cache de arquivos das aplicaes CSDK no cliente. O
 * cache pode ser desabilitado atravs da preferncia
 * {@link DesktopPref#CSDK_CACHE_ENABLED}. O diretrio local utilizado como base
 * pelo cache pode ser configurado pela preferncia
 * {@link DesktopPref#CSDK_CACHE_LOCATION}.
 */
public class ApplicationCache {

  /**
   * Indica se a verso da aplicao j foi carregada no cache.
   *
   * @param registry o registro da aplicao.
   * @return {@code true} se a aplicao j est em cache ou
   * {@code false} caso contrrio.
   */
  public static boolean isApplicationVersionInCache(
    ApplicationRegistry registry) {
    FileInfo[] libInfos = registry.getApplicationLibs();
    if (libInfos != null) {
      File cacheDir = getCachePathForApplicationVersion(registry);
      if (!cacheDir.exists()) {
        return false;
      }
      for (FileInfo libInfo : libInfos) {
        String fileName = libInfo.getName();
        File localFile = new File(cacheDir, fileName);
        if (!localFile.exists()) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Limpa o diretrio local de cache das bibliotecas da aplicao especificada.
   *
   * @param registry o registro da aplicao.
   * @throws ApplicationException em caso de erro ao limpar o cache.
   */
  public static void clearCacheForApplication(
    ApplicationRegistry registry) throws ApplicationException {
    File cacheDir = getCacheBasePathForApplication(registry);
    if (cacheDir.exists()) {
      try {
        if (!FileUtils.delete(cacheDir)) {
          String errorMsg =
            LNG.get("application.manager.error.clear.app.cache");
          throw new ApplicationException(errorMsg);
        }
      }
      catch (Exception e) {
        String errorMsg = LNG.get("application.manager.error.clear.app.cache",
          registry.getId());
        throw new ApplicationException(errorMsg);
      }
    }
  }

  /**
   * Limpa o diretrio local de cache das bibliotecas de todas as aplicaes.
   *
   * @throws ApplicationException em caso de erro ao limpar o cache.
   */
  public static void clearCache() throws ApplicationException {
    File cacheDir = getCacheBasePath();
    if (cacheDir.exists()) {
      try {
        if (!FileUtils.delete(cacheDir)) {
          String errorMsg = LNG.get("application.manager.error.clear.cache");
          throw new ApplicationException(errorMsg);
        }
      }
      catch (Exception e) {
        String errorMsg = LNG.get("application.manager.error.clear.cache");
        throw new ApplicationException(errorMsg);
      }
    }
  }

  /**
   * Obtm o diretrio de cache da aplicao, j preenchido com os arquivos
   * necessrios para a aplicao.
   *
   * @param registry o registro da aplicao.
   * @return o diretrio de cache.
   * @throws ApplicationException caso haja erro na preparao do cache.
   */
  public static File getApplicationCache(
    ApplicationRegistry registry) throws ApplicationException {
    FileInfo[] libs = registry.getApplicationLibs();
    if (libs == null) {
      return null;
    }
    File cacheDir = getCachePathForApplicationVersion(registry);
    if (!isCacheEnabled() || !isApplicationVersionInCache(registry)) {
      fillApplicationCache(registry, cacheDir, libs);
    }
    return cacheDir;
  }

  /**
   * Indica se o cache de aplicaes est habilitado.
   *
   * @return {@code true} caso o cache esteja habilitado ou
   * {@code false} caso contrrio.
   */
  private static boolean isCacheEnabled() {
    PreferenceManager manager = PreferenceManager.getInstance();
    PreferenceCategory allPrefs = manager.loadPreferences();
    PreferenceCategory category = allPrefs.getCategory(DesktopPref.class);
    PVBoolean enabledProperty =
      category.getPVBoolean(DesktopPref.CSDK_CACHE_ENABLED);
    Boolean cacheEnabled = enabledProperty.getValue();
    return cacheEnabled;
  }

  /**
   * Obtm o diretrio de cache de uma verso especfica da aplicao. A verso
   * nesse caso  definida pelo timestamp do registro e no pela verso
   * configurada nas propriedades da aplicao.
   *
   * @param registry o registro da aplicao.
   * @return o diretrio de cache da verso.
   */
  private static File getCachePathForApplicationVersion(
    ApplicationRegistry registry) {
    File applicationDir = getCacheBasePathForApplication(registry);
    return new File(applicationDir, String.valueOf(registry.getTimestamp()));
  }

  /**
   * Obtm o diretrio base de cache onde so armazenados os arquivos de uma
   * aplicao especfica localmente.
   *
   * @param registry o registro da aplicao.
   * @return o diretrio base.
   */
  private static File getCacheBasePathForApplication(
    ApplicationRegistry registry) {
    File baseDir = getCacheBasePath();
    return new File(baseDir, registry.getId());
  }

  /**
   * Obtm o diretrio base de cache onde so armazenados os arquivos das
   * aplicaes localmente.
   *
   * @return o diretrio base.
   */
  private static File getCacheBasePath() {
    PreferenceManager manager = PreferenceManager.getInstance();
    PreferenceCategory allPrefs = manager.loadPreferences();
    PreferenceCategory category = allPrefs.getCategory(DesktopPref.class);
    PVString cacheLocationPref =
      category.getPVString(DesktopPref.CSDK_CACHE_LOCATION);
    String cacheLocationValue = cacheLocationPref.getValue();
    CSDKCacheLocation cacheLocation;
    try {
      cacheLocation = CSDKCacheLocation.valueOf(cacheLocationValue);
    }
    catch (Exception e) {
      String defaultValue = cacheLocationPref.getDefaultValue();
      try {
        cacheLocation = CSDKCacheLocation.valueOf(defaultValue);
      }
      catch (Exception e1) {
        cacheLocation = CSDKCacheLocation.JAVA_IO_TMPDIR;
      }
    }
    File baseDir = cacheLocation.getPath();
    if (baseDir == null || !baseDir.canWrite()) {
      baseDir = CSDKCacheLocation.JAVA_IO_TMPDIR.getPath();
    }

    Client client = Client.getInstance();
    String systemName = client.getSystemName();
    String systemUser = System.getProperty("user.name");
    String systemDirName =
      "." + FileUtils.fixDirectoryName(systemName + "_" + systemUser);
    File systemDir = new File(baseDir, systemDirName);
    String version = client.getVersion();
    String versionDirName = FileUtils.fixDirectoryName(version);
    File versionDir = new File(systemDir, versionDirName);
    User user = User.getLoggedUser();
    String login = user.getLogin();
    String userDirName = FileUtils.fixDirectoryName(login);
    File userDir = new File(versionDir, userDirName);
    return userDir;
  }

  /**
   * Preenche o cache da aplicao, baixando do servidor os arquivos utilizadas
   * pela aplicao especificada.
   *
   * @param registry o registro da aplicao.
   * @param cacheDir o diretrio de cache.
   * @param libInfos dados das biliotecas.
   * @throws ApplicationException em caso de erro ao salvar as bibliotecas.
   */
  private static void fillApplicationCache(ApplicationRegistry registry,
    final File cacheDir,
    final FileInfo[] libInfos) throws ApplicationException {
    final ApplicationServiceInterface applicationService =
      ClientRemoteLocator.applicationService;
    if (applicationService == null) {
      return;
    }
    try {
      clearCacheForApplication(registry);
    }
    catch (Exception e) {
      /*
       * Se o diretrio de cache no tiver sido apagado com sucesso, pode ser
       * que tenham ficado arquivos .nfs pendentes. Podemos seguir e tentar
       * repopular a cache no diretrio existente.
       */
    }
    if (!cacheDir.exists()) {
      if (!cacheDir.mkdirs()) {
        String errorMsg = LNG.get("application.manager.error.lib.dir",
          cacheDir.getPath());
        throw new ApplicationException(errorMsg);
      }
    }
    final String appId = registry.getId();
    final RemoteTask<Void> loadLibTask = new RemoteTask<Void>() {
    	
     @Override
     protected void handleError(Exception error) {
    	 error.printStackTrace();
     }
     
      @Override
      protected void performTask() throws RemoteException {
        for (FileInfo libInfo : libInfos) {
          String[] resoursePath = libInfo.getPathAsArray();
          RemoteFileChannelInfo fileChannel =
            applicationService.getApplicationResource(appId, resoursePath);
          if (fileChannel == null) {
            throw new RemoteException(
              "Application resource not found in server: " + libInfo.getPath());
          }
          try {
            String fileName = libInfo.getName();
            File localFile = new File(cacheDir, fileName);
            downloadFile(fileChannel, localFile);
          }
          catch (final Exception e) {
            throw new RemoteException(
              "Error downloading application resource " + libInfo.getPath(), e);
          }
        }
      }

      /**
       * Traz efetivamente o arquivo para a cahe.
       *
       * @param info o canal para download do arquivo.
       * @param targetFile o arquivo local.
       * @throws Exception em caso de erro ao fazer o download.
       */
      private void downloadFile(RemoteFileChannelInfo info,
        File targetFile) throws Exception {
        SyncRemoteFileChannel channel = null;
        try {
          channel = new SyncRemoteFileChannel(info);
          channel.open(true);
          long size = channel.getSize();
          try (OutputStream target = new FileOutputStream(targetFile)) {
            channel.syncTransferTo(0, size, target);
            target.flush();
          }
        } finally {
          if (channel != null && channel.isOpen()) {
            channel.close();
          }
        }
      }
    };
    
    
    String message =
      LNG.get("application.manager.task.lib.download", appId);
    if (!loadLibTask.execute(null, null, message)) {
      String errorMsg = LNG
        .get("application.manager.error.lib.download", appId);
      throw new ApplicationException(errorMsg, loadLibTask.getError());
    }
  }

}
