package csbase.client.applicationmanager.resourcehelpers;

import java.awt.Window;
import java.util.List;

import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.applications.ApplicationProject;
import csbase.client.desktop.DesktopPref;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.types.PVMap;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileType;
import csdk.v1_0.api.application.IMessage;
import tecgraf.javautils.core.lng.LNG;

/**
 * Classe auxiliar que disponibiliza operaes que envolvem aplicaes
 * preferenciais.
 *
 * @author Tecgraf
 */
public class ApplicationUtils {

  /**
   * Referncia para o gerenciador de preferncias
   */
  private static PreferenceManager pm = PreferenceManager.getInstance();

  /**
   * Referncia para o gerenciador de aplicao
   */
  private static ApplicationManager am = ApplicationManager.getInstance();

  /**
   * Retorna o identificador da aplicao preferencial do usurio que trata o
   * determinado tipo de arquivo.
   *
   * @param owner janela me.
   * @param file arquivo.
   * @return identificador da aplicao preferencial.
   */
  public static String getPreferredApp(Window owner, ClientProjectFile file) {
    ProjectFileType fileType = ProjectFileType.getFileType(file.getType());
    String appId = getPreferredAppFromFileType(owner, fileType);
    return appId;
  }

  /**
   * Retorna o identificador da aplicao preferencial do usurio que trata o
   * determinado tipo de arquivo.
   *
   * @param owner janela me
   * @param fileType tipo do arquivo.
   * @return identificador da aplicao preferencial.
   */
  public static String getPreferredAppFromFileType(Window owner,
    ProjectFileType fileType) {
    String fileTypeCode = fileType.getCode();
    List<String> apps = am.getApplicationsFromType(fileTypeCode);

    if (apps == null) {
      String title = getString("unsupported.type.title");
      String fileTypeDesc = fileType.getDescription();
      String err = getString("unsupported.type.msg") + fileTypeDesc;
      StandardErrorDialogs.showErrorDialog(owner, title, err);
      return null;
    }

    if (apps.size() == 1) {
      return apps.get(0);
    }

    PreferenceCategory root = pm.loadPreferences();
    PreferenceCategory pc = root.getCategory(DesktopPref.class);

    PVMap pv = (PVMap) pc.getPreference(DesktopPref.FILE_TYPE_TO_APP);
    String appId = pv.get(fileTypeCode);

    if (appId == null) {
      PreferredAppDialog dialog = new PreferredAppDialog(owner, apps,
        fileTypeCode, true);
      dialog.setVisible(true);
      appId = dialog.getSelectedApp();

      if (appId != null && dialog.toSave()) {
        pv.put(fileTypeCode, appId);
        PreferenceManager.getInstance().savePreferences();
      }
    }

    return appId;
  }

  /**
   * Mtodo que executa a aplicao preferencial que o usurio escolheu para um
   * determinado tipo de arquivo.
   *
   * @param owner janela me.
   * @param file arquivo.
   * @throws ApplicationException caso ocorra erro na instanciao da aplicao.
   */
  public static void runPreferredApp(Window owner, ClientProjectFile file)
    throws ApplicationException {
    if (file == null) {
      throw new IllegalArgumentException("file no pode ser nulo.");
    }

    String appId = getPreferredApp(owner, file);
    runApp(appId, owner, file);
  }

  /**
   * Mtodo que executa uma aplicao passando um determinado arquivo.
   *
   * @param appId identificador da aplicao.
   * @param owner janela me.
   * @param file arquivo.
   * @throws ApplicationException caso ocorra erro na instanciao da aplicao.
   */
  public static void runApp(String appId, Window owner, ClientProjectFile file)
    throws ApplicationException {
    if (file == null) {
      throw new IllegalArgumentException("file no pode ser nulo.");
    }

    if (appId != null) {
      ApplicationType app = ApplicationManager.getInstance().runApplication(
        appId);
      if (app != null) {
        /*
         * Aplicaes CSDK no podem receber diretamente um ClientProjectFile,
         * pois no tm acesso a essa classe. O ideal  que as aplicaes no
         * troquem objetos complexos como mensagens. Depois que as aplicaes
         * forem migradas para esse novo padro, o
         * ApplicationProject.PROJECT_FILE_MESSAGE pode ser removido.
         */
        if (app instanceof csbase.client.csdk.v1_0.application.CSDKApplication
          || app instanceof csbase.client.csdk.v2.application.CSDKApplication) {
          app.sendMessage(IMessage.PROJECT_FILE_PATH_MESSAGE, file.getPath(),
            null);
        }
        else {
          app.sendMessage(ApplicationProject.PROJECT_FILE_MESSAGE, file, null);
        }
      }
    }
  }

  /**
   * Mtodo auxiliar que obtem a string internacionalizada.
   *
   * @param key chave da string.
   * @return string internacionalizada.
   */
  private static String getString(String key) {
    return LNG.get("ApplicationUtils." + key);
  }

}
