package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;

import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.AlgorithmsManagerAdapter;
import csbase.client.applications.algorithmsmanager.actions.CommonManagementAction;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.client.util.ClientUtilities;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;

/**
 * Classe abstrata que define um painel comum para funcionalidades de seleo de
 * dados e definio das operaes que podem ser realizadas sobre esses dados
 * selecionados. Esse painel precisa ser definido tanto na funcionalidade de
 * gerenciamento de categorias quanto na funcionalidade de gerenciamento de
 * algoritmos, na aplicao Gerenciador de Algoritmos.
 * 
 * As operaes so estabelecidas redefinindo-se o mtodo
 * <code> setOperationActions </code>. Permite-se no passar nenhuma ao. Para
 * cada ao definida,  criado um boto, que  adicionado na mesma ordem
 * listada no conjunto de aes desse painel.
 * 
 * O painel para seleo dos dados deve ser construdo redefinindo-se o mtodo
 * <code> buildMainSelectionPanel </code>.
 */
public abstract class CommonSelectionPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> {

  /** Ao de gerenciamento que criou o painel */
  private CommonManagementAction action;

  /**
   * Lista com as aes correspondentes a cada operao que pode ser realizada
   * sobre os dados selecionados.
   */
  private List<Action> operationActions;

  /**
   * Constri o painel.
   * 
   * @param action ao que criou o dilogo
   */
  public CommonSelectionPanel(CommonManagementAction action) {
    super(action.getApplication());
    this.action = action;
    this.operationActions = buildOperationActions();
    buildPanel();
    action.getApplication().addAlgorithmsManagerListener(
      new AlgorithmsManagerAdapter() {

        @Override
        public void categoryUpdated(CategorySet modifiedCategorySet) {
          handleCategoryUpdated(modifiedCategorySet);
        }

        @Override
        public void categoryRemoved(Category category) {
          handleCategoryRemoved(category);
        }

        @Override
        public void categoryCreated(Category category) {
          handleCategoryCreated(category);
        }

        @Override
        public void algorithmCreated(AlgorithmInfo algoInfo) {
          handleAlgorithmCreated(algoInfo);
        }

        @Override
        public void algorithmRemoved(AlgorithmInfo algoInfo) {
          handleAlgorithmRemoved(algoInfo);
        }

        @Override
        public void algorithmUpdated(AlgorithmInfo algoInfo) {
          handleAlgorithmUpdated(algoInfo);
        }
      });
  }

  /**
   * Realiza uma ao quando uma categoria  criada na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param category categoria criada
   */
  protected void handleCategoryCreated(Category category) {
  }

  /**
   * Realiza uma ao quando uma categoria  removida na aplicao Gerenciador
   * de Algoritmos.
   * 
   * @param category categoria removida
   */
  protected void handleCategoryRemoved(Category category) {
  }

  /**
   * Realiza uma ao quando um conjunto de categorias  alterado na aplicao
   * Gerenciador de Algoritmos.
   * 
   * @param modifiedCategorySet conjunto de categorias modificadas
   */
  protected void handleCategoryUpdated(CategorySet modifiedCategorySet) {
  }

  /**
   * Realiza uma ao quando um algoritmo  alterado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo alterado
   */
  protected void handleAlgorithmUpdated(AlgorithmInfo algoInfo) {
  }

  /**
   * Realiza uma ao quando um algoritmo  criado na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo criado
   */
  protected void handleAlgorithmCreated(AlgorithmInfo algoInfo) {
  }

  /**
   * Realiza uma ao quando um algoritmo  removido na aplicao Gerenciador de
   * Algoritmos.
   * 
   * @param algoInfo informaes do algoritmo removido
   */
  protected void handleAlgorithmRemoved(AlgorithmInfo algoInfo) {
  }

  /**
   * Constri o painel principal para seleo dos dados.
   * 
   * @return o painel principal de seleo criado
   * 
   */
  protected abstract JPanel buildMainSelectionPanel();

  /**
   * Estabelece as aes que representam as operaes que podem ser realizadas
   * sobre os dados selecionados.
   * 
   * @return a lista de aes correspondentes a cada operao que pode ser
   *         realizada sobre os dados selecionados
   * 
   */
  protected abstract List<Action> buildOperationActions();

  /**
   * Obtm um lista com os dados selecionados nesse painel.
   * 
   * @return uma lista com os dados selecionados
   */
  public abstract List<DataInterface> getSelectedDataList();

  /**
   * Confirma se foi possvel realizar a mudana para o novo dado selecionado.
   * 
   * Por exemplo, se o dado corrente estiver em modo de edio, ento o usurio
   * deve confirmar se deseja cancelar ou no, antes de mudar para a nova
   * seleo.
   * 
   * @return retorna true, se a mudana para o novo dado pode ser realizada,
   *         caso contrrio, retorna false
   */
  protected abstract boolean confirmSelectionChanged();

  /**
   * Obtm o primeiro dado selecionado nesse painel.
   * 
   * @return o dado selecionado, caso contrrio, retorna null
   */
  public DataInterface getSelectedData() {
    List<DataInterface> selectedDataList = getSelectedDataList();
    if (selectedDataList != null && !selectedDataList.isEmpty()) {
      return selectedDataList.get(0);
    }
    return null;
  }

  /**
   * Obtm a ao de gerenciamento dos dados que criou o painel.
   * 
   * @return a ao de gerenciamento dos dados
   * 
   */
  protected CommonManagementAction getManagementAction() {
    return action;
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.applications.AbstractSimpleApplicationPanel#buildPanel()
   */
  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());
    JPanel mainPanel = buildMainSelectionPanel();
    JPanel operationPanel = getOperationPanel();
    add(mainPanel, new GBC(0, 0).both().insets(5, 5, 5, 5));
    add(operationPanel,
      new GBC(0, 1).horizontal().southwest().insets(5, 5, 5, 5));
  }

  /**
   * Cria o painel com os botes referentes todas as operaes que podem ser
   * realizadas sobre os dados selecionados. Os botes so criados a partir das
   * aes passadas para a construo do painel, e so adicionados lado a lado,
   * na mesma ordem em que so lidos da lista.
   * 
   * @return o painel criado
   */
  private JPanel getOperationPanel() {
    JPanel operationPanel = new JPanel(new GridBagLayout());
    List<JComponent> buttons = buildButtons();

    int x = 0, y = 0;
    for (JComponent button : buttons) {
      operationPanel.add(button,
        new GBC(x++, y).none().west().insets(5, 5, 5, 5));
    }
    ClientUtilities.adjustEqualSizes(buttons.toArray(new JComponent[0]));
    return operationPanel;
  }

  /**
   * Constri todos os botes correspondentes s aes passadas para cada
   * operao que pode ser realizada sobre o dado selecionado.
   * 
   * @return um array com os botes
   */
  private List<JComponent> buildButtons() {
    List<JComponent> buttons = new ArrayList<>();
    for (Action opAction : operationActions) {
      buttons.add(getOperationButton(opAction));
    }
    return buttons;
  }

  /**
   * Obtm um boto a partir de sua ao correspondente.
   * 
   * @param opAction ao associada ao boto
   * 
   * @return o boto correspondente a ao
   */
  private JComponent getOperationButton(Action opAction) {
    JButton button = new JButton(opAction);
    button.setToolTipText((String) opAction.getValue(Action.SHORT_DESCRIPTION));
    return button;
  }

  /**
   * Habilita ou desabilita as aes dos botes que representam as operaes
   * sobre os dados selecionados.
   * 
   * @param opAction ao da operao que vai modificar seu estado
   * @param state se true, habilita as aes, caso contrrio, desabilita as
   *        aes
   */
  protected void enableAction(Action opAction, boolean state) {
    opAction.setEnabled(state);
  }
}
