package csbase.client.applications.flowapplication.graph.utils;

import java.awt.Dimension;
import java.awt.Window;
import java.awt.geom.Dimension2D;
import java.awt.geom.Point2D;

import csbase.client.algorithms.AlgorithmConfiguratorFactory;
import csbase.client.algorithms.AlgorithmConfiguratorView;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.validation.ValidationMode;

/**
 * Operao de atualizao de verso do algoritmo de um n de um fluxo.
 *
 * @author isabella
 */
public class GraphNodeUpdateOperation {

  /**
   * O n a ser atualizado.
   */
  private GraphNode node;

  /**
   * Construtor
   *
   * @param node O n a ser atualizado.
   */
  public GraphNodeUpdateOperation(GraphNode node) {
    if (node == null) {
      throw new IllegalArgumentException("O parmetro node est nulo!");
    }
    this.node = node;
  }

  /**
   * Determina se existe uma verso mais nova para o algoritmo que um
   * determinado n do grafo representa.
   *
   * @param node N que dever ser testado
   * @return Verdadeiro se o n representa um algoritmo desatualizado
   */
  public static boolean hasNewerVersion(GraphNode node) {
    AlgorithmVersionId nodeVersionId = node.getAlgorithmVersionId();
    AlgorithmInfo algorithm = node.getAlgorithm();
    if (algorithm != null) {
      AlgorithmVersionInfo currentVersion = algorithm.getVersionInfo(
        nodeVersionId);
      AlgorithmVersionInfo lastVersion = algorithm.getLastVersion();
      if (lastVersion != null && currentVersion != null) {
        AlgorithmVersionId latestVersionId = lastVersion.getId();
        if (!nodeVersionId.equals(latestVersionId) && currentVersion
          .getType() == lastVersion.getType()) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Determina se existe uma verso mais nova para o algoritmo que o n
   * representa.
   *
   * @return Verdadeiro se o n representa um algoritmo desatualizado.
   */
  public boolean hasNewerVersion() {
    return hasNewerVersion(node);
  }

  /**
   * Atualiza o n que representa um algoritmo desatualizado (que tem pelo menos
   * uma verso mais nova disponvel). A janela de configurao de parmetros
   * pode ser mostrada aps a atualizao.
   *
   * @param askForParameters Se verdadeiro, a janela de configurao de
   *        parmetros ser aberta aps a atualizao no caso de avisos ou
   *        erros.
   * @return verdadeiro se a atualizao do n foi feita com sucesso ou falso,
   *         caso contrrio.
   */
  public boolean update(boolean askForParameters) {
    return updateNode(askForParameters);
  }

  /**
   * Atualiza o n que representa um algoritmo desatualizado (que tem pelo menos
   * uma verso mais nova disponvel). A janela de configurao de parmetros do
   * n  mostrada aps a atualizao.
   *
   * @return verdadeiro se a atualizao do n foi feita com sucesso ou falso,
   *         caso contrrio.
   */
  public boolean update() {
    return update(true);
  }

  /**
   * Atualiza um determinado n que representa um algoritmo desatualizado (que
   * tem pelo menos uma verso mais nova disponvel). A janela de configurao
   * de parmetros do n  mostrada aps a atualizao no caso de avisos ou
   * erros se askForParameters for verdadeiro.
   *
   * @param askForParameters Se verdadeiro, a janela de configurao de
   *        parmetros ser aberta aps a atualizao no caso de avisos ou erros
   * @return verdadeiro se a atualizao do n foi feita com sucesso ou falso,
   *         caso contrrio.
   */
  protected boolean updateNode(boolean askForParameters) {
    boolean result = true;
    if (hasNewerVersion()) {
      AlgorithmInfo algorithmInfo = node.getAlgorithm();
      AlgorithmVersionInfo algorithmVersionInfo = algorithmInfo
        .getLastVersion();
      AlgorithmVersionInfo currentVersion = algorithmInfo.getVersionInfo(node
        .getAlgorithmVersionId());
      if (algorithmVersionInfo.getType() != currentVersion.getType()) {
        return false;
      }
      Graph graph = node.getGraph();
      Window window = graph.getParentWindow();
      AlgorithmConfiguratorView algorithmConfiguratorView =
        AlgorithmConfiguratorFactory.getInstance().createConfigurationView(
          window, algorithmVersionInfo, ValidationMode.ALLOW_EMPY_VALUES);
      if (algorithmConfiguratorView == null) {
        return true;
      }
      Point2D nodeLocation = new Point2D.Double(Math.max(0, node.getX()), Math
        .max(0, node.getY()));
      Dimension2D dimension = new Dimension();
      dimension.setSize(node.getWidth(), node.getHeight());
      GraphNode newNode = graph.createGraphNode(algorithmConfiguratorView, node
        .getLabel(), nodeLocation, dimension, node.isBypassed());
      CopyAndPasteOperation operation = new CopyAndPasteOperation(node, newNode,
        true);
      OperationStatus operationStatus = operation.execute();
      if (operationStatus.isConfirmed()) {
        node.deattach();
        // Mantm o ID do n para facilitar a manuteno do fluxo
        newNode.setId(node.getId());
        if (askForParameters && (operationStatus.hasWarnings()
          || operationStatus.hasErrors())) {
          newNode.askForParameterValues();
        }
      }
      else {
        //rollback
        newNode.deattach();
      }
    }
    return result;
  }
}
