package csbase.client.applications.preferenceviewer;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JSeparator;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationEditPrefAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferencePolicy;

/**
 * Aplicao que permite que o usurio visualize/edite todas as suas
 * preferncias.
 * 
 * Essa aplicao tambm pode ser chamada via outra aplicao e, com isso,
 * editar apenas as preferncias do usurio de uma aplicao especfica. Para
 * isso, basta usarmos a ao {@link ApplicationEditPrefAction}.
 * 
 * Essa aplicao exibe a rvore de categorias no lado esquerdo.  possvel
 * definirmos a ordem que as categorias so exibidas, para isso, cada sistema
 * deve definir uma propriedade com nome 'tree.order', do tipo lista, em seu
 * arquivo de propriedades.
 * 
 * @author Tecgraf
 */
public class PreferenceViewer extends Application {

  /** Painel que exibe todas as preferncias. */
  private PreferencePanel mainPanel;

  /**
   * Construtor padro.
   * 
   * @param id - identificador da aplicao.
   */
  public PreferenceViewer(String id) {
    super(id);

    mainPanel = new PreferencePanel(this);

    buildInterface();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
    mainPanel.removeAllListeners();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(String name, Object value, String senderId) {
    if (Application.PREFERENCE_MESSAGE.equals(name) && value != null) {
      mainPanel.rebuidInterface((PreferenceCategory) value);
    }
  }

  /** Constroi a interface da aplicao. */
  private void buildInterface() {
    ApplicationFrame appFrame = getApplicationFrame();
    appFrame.setLayout(new GridBagLayout());

    JButton ok = createOkButton();
    JButton cancel = createCancelButton();

    JRootPane rootPane = appFrame.getRootPane();
    rootPane.setDefaultButton(ok);

    GUIUtils.matchPreferredSizes(new JComponent[] { ok, cancel,
        mainPanel.getRestoreButton() });

    JPanel buttonsPanel = new JPanel(new GridBagLayout());
    buttonsPanel.add(new JLabel(), new GBC(1, 0).horizontal());
    buttonsPanel.add(cancel, new GBC(2, 0).insets(5).east());
    buttonsPanel.add(ok, new GBC(3, 0).insets(5).east());

    appFrame.add(mainPanel, new GBC(0, 0).both().left(10).right(10).top(10));
    appFrame.add(new JSeparator(JSeparator.HORIZONTAL), new GBC(0, 1)
      .horizontal());
    appFrame.add(buttonsPanel, new GBC(0, 2).horizontal().left(10).right(10));
    appFrame.pack();
  }

  /**
   * Cria o boto 'Cancelar' da aplicao.
   * 
   * @return boto 'Cancelar'.
   */
  private JButton createCancelButton() {
    JButton cancel = new JButton(getString("button.cancel"));

    cancel.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        closeApplication();
      }
    });

    return cancel;
  }

  /**
   * Cria o boto 'Ok' da aplicao.
   * 
   * @param <T> - tipo do valor de preferncia usado para fazer casting durante
   *        a iterao.
   * 
   * @return boto 'Ok'.
   */
  @SuppressWarnings("unchecked")
  private <T> JButton createOkButton() {
    JButton ok = new JButton("Ok");

    ok.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        for (PreferenceCategory pc : mainPanel.getModifiedCategories()) {
          for (PreferenceValue<?> pv : pc.getPreferences(
            PreferencePolicy.READ_ONLY, PreferencePolicy.READ_WRITE)) {
            pv.getEditor().commit();
          }

          if (pc.isCopy() && mainPanel.saveLocalPreferences()) {
            PreferenceCategory originalPc = pc.getOriginal();
            for (PreferenceValue<?> pv : pc.getPreferences(
              PreferencePolicy.READ_ONLY, PreferencePolicy.READ_WRITE)) {

              PreferenceValue<T> copyPv = (PreferenceValue<T>) pv;
              PreferenceValue<T> originalPv =
                (PreferenceValue<T>) originalPc.getPreference(copyPv.getName());
              originalPv.setValue(copyPv.getValue());
            }
          }
        }

        PreferenceManager.getInstance().savePreferences(getApplicationFrame());
        closeApplication();
      }
    });

    return ok;
  }
}
