package csbase.client.applications.projectsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.ProjectSharingInfoPanel;
import csbase.client.applications.projectsmanager.proxy.AdminSendEmailTask;

/**
 * Dilogo de envio de emails referentes a projetos.
 * 
 * @author jnlopes
 */
public class AdminMailDialog extends ProjectsManagerDialog {
  /**
   * Lista de projetos selecionados.
   */
  private List<ProjectsManagerData> projectsManagerData;

  /**
   * Campo de texto onde o "Assunto" deve ser preenchido.
   */
  private final JTextField subjectField = new JTextField();

  /**
   * rea de texto onde o mail deve ser redigido.
   */
  private final JTextArea mailArea = new JTextArea();

  private final JCheckBox sendCopyBox = new JCheckBox();

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao.
   * @param projectsManagerData Lista de projetos selecionados.
   */
  public AdminMailDialog(final ProjectsManager projectsManager,
    final List<ProjectsManagerData> projectsManagerData) {
    super(projectsManager);
    this.projectsManagerData = projectsManagerData;
    buildPanel();
    setTitle(getString("AdminMailDialog.title"));
  }

  /**
   * Monta o painel com os botes.
   * 
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {
    final JButton sendButton = new JButton();
    final JButton cancelButton = new JButton();

    final AdminMailDialog amd = this;
    sendButton.setAction(new AbstractAction() {
      /**
       * {@inheritDoc}
       */
      @Override
      public void actionPerformed(ActionEvent e) {
        final ProjectsManager projectsManager = getProjectsManager();
        final Map<Object, Set<String>> userProjectMap =
          generateUserProjectMap();

        final AdminSendEmailTask aset =
          new AdminSendEmailTask(projectsManager, userProjectMap, subjectField
            .getText(), mailArea.getText(), sendCopyBox.isSelected());

        final ApplicationFrame appFrame = projectsManager.getApplicationFrame();
        final String taskTitle = projectsManager.getName();
        final String taskMsg = getString("AdminMailDialog.running.message");
        aset.execute(appFrame, taskTitle, taskMsg);

        if (aset.wasCancelled()) {
          final String err = getString("AdminMailDialog.cancelled.message");
          aset.showError(err);
          return;
        }

        if (aset.getStatus() != true) {
          final Exception exception = aset.getError();
          final String err =
            getString("AdminMailDialog.send.mail.error.message");
          final StringBuilder strBuilder = new StringBuilder(err);
          strBuilder.append("\n");
          strBuilder.append(exception.getMessage());
          StandardDialogs.showErrorDialog(appFrame, "", strBuilder.toString());
        }

        final String[] ret = aset.getResult();
        if (ret != null) {
          if (ret.length == 0) {
            final String errMsg = getString("AdminMailDialog.user.mail.error");
            StandardDialogs.showErrorDialog(appFrame, "", errMsg);
            return;
          }

          final StringBuilder users = new StringBuilder();
          for (String str : ret) {
            users.append(" - " + str + "\n");
          }
          final String err =
            getString("AdminMailDialog.failure.sending.mail.message");
          final String errMsg =
            getString("AdminMailDialog.failure.sending.mail.complement.message");
          final StringBuilder msg = new StringBuilder(err);
          msg.append("\n" + users.toString() + "\n" + errMsg);
          StandardDialogs.showErrorDialog(appFrame, projectsManager.getName(),
            msg.toString());
        }
        amd.dispose();
      }
    });
    sendButton.setText(getString("AdminMailDialog.send.text"));

    cancelButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        amd.dispose();
      }
    });
    cancelButton.setText(getString("AdminMailDialog.cancel.text"));

    final JComponent[] buttons = { sendButton, cancelButton };
    GUIUtils.matchPreferredSizes(buttons);

    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(sendButton);
    buttonsPanel.add(cancelButton);
    return buttonsPanel;
  }

  /**
   * Gera o HashMap contendo a relao (1-N) Usurio - Projetos.
   * 
   * @return HashMap contendo a relao (1-N) Usurio - Projetos.
   */
  private Map<Object, Set<String>> generateUserProjectMap() {
    Map<Object, Set<String>> userProjectMap =
      new HashMap<Object, Set<String>>();
    for (ProjectsManagerData pmd : projectsManagerData) {
      Set<String> aux = userProjectMap.get(pmd.getOwnerId());
      if (aux == null) {
        aux = new HashSet<String>();
      }
      aux.add(pmd.getProjectName());
      userProjectMap.put(pmd.getOwnerId(), aux);
    }
    return userProjectMap;
  }

  /**
   * Monta o componente de insero do texto.
   * 
   * @return o componente
   */
  private JScrollPane buildTextPane() {
    mailArea.setRows(5);
    mailArea.setColumns(40);
    mailArea.setLineWrap(true);

    final JScrollPane pane = new JScrollPane(mailArea);

    return pane;
  }

  /**
   * Monta o componente dos projetos.
   * 
   * @return o componente
   */
  private ProjectSharingInfoPanel buildProjectPanel() {
    final ProjectsManager projectsManager = getProjectsManager();
    final ProjectSharingInfoPanel projectsPanel =
      new ProjectSharingInfoPanel(projectsManager, projectsManagerData, false,
        true);

    return projectsPanel;
  }

  /**
   * Monta o painel.
   */
  private void buildPanel() {
    final ProjectSharingInfoPanel projectsPanel = buildProjectPanel();
    final JPanel subjectPanel = buildSubjectPanel();
    final JScrollPane textPanel = buildTextPane();
    final JPanel buttonsPanel = buildButtonsPanel();

    JLabel projectsLabel =
      new JLabel(getString("AdminMailDialog.projects.title"));
    JLabel subjectLabel =
      new JLabel(getString("AdminMailDialog.subject.title"));
    JLabel textLabel = new JLabel(getString("AdminMailDialog.mail.area.title"));

    sendCopyBox.setText(getString("AdminMailDialog.send.copy.box"));

    int RL = 11;
    int T = 3;
    int B = 9;

    setLayout(new GridBagLayout());
    add(projectsLabel, new GBC(0, 0).none().insets(B, RL, 0, RL).west());
    add(projectsPanel, new GBC(0, 1).horizontal().insets(T, RL, B, RL));

    add(subjectLabel, new GBC(0, 2).none().insets(T, RL, 0, RL).west());
    add(subjectPanel, new GBC(0, 3).horizontal().insets(T, RL, B, RL));

    add(textLabel, new GBC(0, 4).none().insets(T, RL, 0, RL).west());
    add(textPanel, new GBC(0, 5).both().insets(T, RL, B, RL));

    add(sendCopyBox, new GBC(0, 6).both().insets(T, RL, B, RL));

    add(buttonsPanel, new GBC(0, 7).horizontal().insets(T, RL, B, RL));
  }

  /**
   * Monta painel referente ao campo de "Assunto".
   * 
   * @return O painel referenter ao campo de "Assunto".
   */
  private JPanel buildSubjectPanel() {

    final JPanel panel = new JPanel(new GridBagLayout());
    panel.add(subjectField, new GBC(0, 0).horizontal());

    return panel;
  }
}
