package csbase.client.applications.projectsmanager.panels.filters;

import java.awt.GridBagLayout;
import java.awt.event.KeyListener;
import java.util.List;
import java.util.regex.Pattern;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;

/**
 * Painel que contm o filtro por localizao do projeto.
 * 
 * @author jnlopes
 */
public class ProjectFileSystemLocationPanel extends AbstractProjectFilter {

  /**
   * Campo de filesystem de projeto.
   */
  private JTextField projectLocationField;

  /**
   * Label de filesystem de projeto.
   */
  private JLabel projectLocationLabel;

  /**
   * Painel que contm o label da localizao do projeto.
   */
  private JPanel labelPanel;

  /**
   * Painel que contm o campo de texto da localizao do projeto.
   */
  private JPanel textPanel;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public ProjectFileSystemLocationPanel(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * Inicializa componentes.
   */
  @Override
  protected void initComponents() {
    projectLocationField = new JTextField();
    projectLocationLabel =
      new JLabel(
        getString("ProjectFileSystemLocationPanel.project.location.label"));
    labelPanel = new JPanel(new GridBagLayout());
    textPanel = new JPanel(new GridBagLayout());
    setLayout(new GridBagLayout());
  }

  /**
   * Monta o painel.
   */
  @Override
  protected void buildPanel() {
    labelPanel.add(projectLocationLabel, new GBC(0, 0));
    textPanel.add(projectLocationField, new GBC(1, 0).horizontal());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void runSpecificFilter(final List<ProjectsManagerData> projects,
    final List<ProjectsManagerData> filteredProjects) {

    final Pattern pattern = convertToPattern(projectLocationField);
    if (pattern == null) {
      return;
    }

    for (ProjectsManagerData pmd : projects) {
      if (projectMatchesFilter(pattern, pmd)) {
        filteredProjects.add(pmd);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean projectMatchesFilter(final ProjectsManagerData prj) {
    return projectMatchesFilter(convertToPattern(projectLocationField), prj);
  }

  /**
   * Verifica se um projeto  aceito por este filtro.
   * 
   * @param pattern padro
   * @param prj projeto
   * @return <code>true</code> se o projeto  aceito por este filtro
   */
  private boolean projectMatchesFilter(final Pattern pattern,
    final ProjectsManagerData prj) {
    if (pattern == null) {
      return false;
    }
    final String prjLocation = prj.getLocationInServer();
    if (prjLocation == null || prjLocation.isEmpty()) {
      return false;
    }
    return pattern.matcher(prjLocation).matches();
  }

  /**
   * Adiciona key listener padro aos elementos deste painel que devem disparar
   * uma nova filtragem sempre que forem modificados.
   * 
   * @param kl Key listener
   */
  @Override
  public void addListener(KeyListener kl) {
    this.projectLocationField.addKeyListener(kl);
  }

  /**
   * Remove qualquer informao presente nos campos de filtro.
   */
  @Override
  public void emptyFilterData() {
    this.projectLocationField.setText("");
  }

  /**
   * Retorna o painel do label da localizao do projeto.
   * 
   * @return O painel do label da localizao do projeto.
   */
  public JPanel getLabelPanel() {
    return labelPanel;
  }

  /**
   * Retorna o painel de campo de texto da localizao do projeto.
   * 
   * @return O painel de campo de texto da localizao do projeto.
   */
  public JPanel getTextPanel() {
    return textPanel;
  }

}
