package csbase.client.csdk.v2.filesystem;

import java.awt.Window;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import csbase.client.ClientLocalFile;
import csbase.client.ClientSmartFile;
import csbase.client.project.ClientFileLock;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ClientProjectFile;
import csdk.v2.api.filesystem.FileLocationType;
import csdk.v2.api.filesystem.FileLockedException;
import csdk.v2.api.filesystem.IFile;
import csdk.v2.api.filesystem.IFileLock;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Encapsula um arquivo (da rea de projetos ou do sistema local) em um arquivo
 * acessvel pelas aplicaes baseadas no CSDK.
 */
public class CSDKFile implements IFile {

  /**
   * Arquivo encapsulado.
   */
  private final ClientFile file;

  /**
   * Construtor
   *
   * @param file arquivo encapsulado.
   */
  public CSDKFile(ClientFile file) {
    if (file == null) {
      throw new IllegalArgumentException("Parmetro file no pode ser nulo");
    }
    this.file = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws IOException {
    file.close(force);
  }

  /**
   * Fecha o arquivo.
   *
   * @throws IOException em caso de erro de I/O.
   */
  public void close() throws IOException {
    file.close(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] getChildren() throws Exception {
    ClientFile[] children = file.getChildren();
    List<IFile> files = new ArrayList<>();
    for (ClientFile clientProjectFile : children) {
      files.add(new CSDKFile(clientProjectFile));
    }
    return files.toArray(new IFile[files.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    return file.getInputStream();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate() {
    return file.getModificationDate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return file.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    return file.getOutputStream();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    return file.getPath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    return file.getStringPath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return file.getType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(boolean readOnly) throws Exception {
    file.open(readOnly);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long position() throws IOException {
    return file.position();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    file.position(newPosition);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len,
    long position) throws Exception {
    return file.read(dst, off, len, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, long position) throws Exception {
    return file.read(dst, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long size() {
    return file.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len,
    long position) throws IOException, FileLockedException {
    file.write(src, off, len, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src,
    long position) throws IOException, FileLockedException {
    file.write(src, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileLocationType getFileLocationType() {
    switch (file.getClientFileType()) {
      case LOCAL:
        return FileLocationType.LOCAL;
      case REMOTE:
        return FileLocationType.REMOTE;
      case SMART:
        return FileLocationType.REMOTE;
      default:
        return null;
    }
  }

  /**
   * Obtm o arquivo encapsulado.
   *
   * @return file o arquivo.
   */
  public ClientFile getFile() {
    return file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getParent() {
    ClientFile parent = file.getParent();
    if (parent == null) {
      return null;
    }
    return new CSDKFile(parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean exists() throws IOException {
    return file.exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return file.canExecute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return file.canRead();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return file.canWrite();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFileLock acquireExclusiveLock(Window window) throws Exception {
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      return new CSDKProjectFileLock(
        ClientFileLock.acquireExclusiveLock(window, projectFile));
    }
    else if (file.getClientFileType() == ClientFileType.SMART) {
      ClientSmartFile smartFile = (ClientSmartFile) file;
      return new CSDKProjectFileLock(ClientFileLock
        .acquireExclusiveLock(window, smartFile.getClientProjectFile()));
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      String[] path = clientlocalFile.getPath();
      File localFile = new File(FileUtils.joinPath(path));
      return new CSDKLocalFileLock(localFile, false, window);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFileLock acquireSharedLock(Window window) throws Exception {
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      return new CSDKProjectFileLock(
        ClientFileLock.acquireSharedLock(window, projectFile));
    }
    else if (file.getClientFileType() == ClientFileType.SMART) {
      ClientSmartFile smartFile = (ClientSmartFile) file;
      return new CSDKProjectFileLock(ClientFileLock
        .acquireSharedLock(window, smartFile.getClientProjectFile()));
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      String[] path = clientlocalFile.getPath();
      File localFile = new File(FileUtils.joinPath(path));
      return new CSDKLocalFileLock(localFile, true, window);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getChild(String name, Window window) throws Exception {
    if (name == null || name.trim().isEmpty()) {
      throw new IllegalArgumentException("Parmetro name no pode ser nulo");
    }

    if(!isDirectory()) {
      throw new IllegalArgumentException(
        "No  possvel listar filhos: caminho no corresponde a diretrio.");
    }

    ClientFile[] files = file.getChildren();
    if (files == null) {
      return null;
    }
    for (ClientFile clientFile : files) {
      if (clientFile.getName().equals(name)) {
        return new CSDKFile(clientFile);
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isHidden() {
    return file.getName().charAt(0) == '.';
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean rename(String newName, Window window) throws Exception {
    if (newName == null) {
      throw new IllegalArgumentException("Parmetro newName no pode ser nulo");
    }
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      projectFile.rename(newName);
      return true;
    }
    else if (file.getClientFileType() == ClientFileType.SMART) {
      ClientSmartFile smartFile = (ClientSmartFile) file;
      smartFile.getClientProjectFile().rename(newName);
      return true;
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      return clientlocalFile.rename(newName);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean move(IFile newParent, Window window) throws Exception {
    if (newParent == null) {
      throw new IllegalArgumentException(
        "Parmetro newParent no pode ser nulo");
    }
    CSDKFile parentDir = (CSDKFile) newParent;
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      if (newParent
        .getFileLocationType() == FileLocationType.REMOTE && newParent
        .isDirectory()) {
        ClientProjectFile parentDirFile =
          (ClientProjectFile) parentDir.getFile();
        projectFile.move(parentDirFile);
        return true;
      }
    }
    else if (file.getClientFileType() == ClientFileType.SMART) {
      ClientSmartFile smartFile = (ClientSmartFile) file;
      if (newParent
        .getFileLocationType() == FileLocationType.REMOTE && newParent
        .isDirectory()) {
        ClientProjectFile parentDirFile =
          (ClientProjectFile) parentDir.getFile();
        smartFile.getClientProjectFile().move(parentDirFile);
        return true;
      }
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      if (newParent.getFileLocationType() == FileLocationType.LOCAL && newParent
        .isDirectory()) {
        ClientLocalFile parentDirFile = (ClientLocalFile) parentDir.getFile();
        return clientlocalFile.move(parentDirFile);
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean delete(Window window) throws Exception {
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      projectFile.remove();
      return true;
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      return clientlocalFile.remove();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile copy(IFile newParent, Window window) throws Exception {
    if (newParent == null) {
      throw new IllegalArgumentException(
        "Parmetro newParent no pode ser nulo");
    }
    CSDKFile parentDir = (CSDKFile) newParent;
    if (isDirectory()) {
      throw new Exception("Diretrio no pode ser copiado!");
    }

    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      return copyRemote(projectFile, parentDir, window);
    }

    if (file.getClientFileType() == ClientFileType.SMART) {
      ClientSmartFile smartFile = (ClientSmartFile) file;
      return copyRemote(smartFile.getClientProjectFile(), parentDir, window);
    }

    if (!newParent.isDirectory()) {
      throw new Exception(
        "Destino no  um diretrio: " + newParent.getStringPath());
    }

    ClientLocalFile clientlocalFile = (ClientLocalFile) file;
    if (newParent.getFileLocationType() == FileLocationType.LOCAL) {
      ClientLocalFile parentDirFile = (ClientLocalFile) parentDir.getFile();
      ClientLocalFile copy = clientlocalFile.copy(parentDirFile);
      if (copy != null) {
        return new CSDKFile(copy);
      }
      return null;
    }
    return null;
  }

  /**
   * Copia o arquivo remoto.
   *
   * @param projectFile arquivo a ser copiado.
   * @param parentDir novo diretrio-pai.
   * @param window janela da operao.
   * @return a cpia do arquivo.
   * @throws Exception em caso de erro.
   */
  private IFile copyRemote(ClientProjectFile projectFile, CSDKFile parentDir,
    Window window) throws Exception {
    if (parentDir.getFileLocationType() == FileLocationType.REMOTE && parentDir
      .isDirectory()) {
      ClientProjectFile parentDirFile = (ClientProjectFile) parentDir.getFile();
      projectFile.copy(parentDirFile);
      return parentDir.getChild(getName(), window);
    }
    return null;
  }

}
