package csbase.client.algorithms;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.io.InputStream;

import javax.swing.AbstractAction;
import javax.swing.Action;

import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.Task;
import csbase.logic.FileInfo;
import csbase.logic.RemoteFileInputStream;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;

/**
 * Ao utilizada para exibir o release notes de uma verso de algoritmo.
 *
 * @author Tecgraf/PUC-Rio.
 */
public class AlgorithmReleaseNotesAction extends AbstractAction {

  /**
   * A verso do algoritmo.
   */
  private AlgorithmVersionInfo versionInfo;

  /**
   * O algoritmo.
   */
  private AlgorithmInfo algorithmInfo;

  /**
   * A janela.
   */
  private final Window window;

  /**
   * Cria uma ao.
   *
   * @param window A janela que est requisitando a ao.
   *
   * @throws IllegalArgumentException Se window for nulo.
   */
  public AlgorithmReleaseNotesAction(Window window) {
    if (window == null) {
      throw new IllegalArgumentException("O parmetro window est nulo.");
    }
    this.window = window;
    showName();
    showIcon();
    putValue(Action.SHORT_DESCRIPTION, LNG.get("algorithms.full_description"));
    setEnabled(false);
  }

  /**
   * Cria uma ao.
   *
   * @param window A janela que est requisitando a ao.
   * @param versionInfo A verso do algoritmo.
   * @param algorithmInfo O algoritmo.
   *
   * @throws IllegalArgumentException Se window ou versionInfo forem nulos.
   */
  public AlgorithmReleaseNotesAction(Window window,
    AlgorithmVersionInfo versionInfo, AlgorithmInfo algorithmInfo) {
    this(window);
    if (versionInfo == null) {
      throw new IllegalArgumentException("O parmetro versionInfo est nulo.");
    }
    setAlgorithmVersionInfo(versionInfo, algorithmInfo);
  }

  /**
   *
   * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
   */
  @Override
  public void actionPerformed(ActionEvent ev) {
    if (versionInfo == null) {
      throw new IllegalStateException("No existe verso corrente, ou seja, "
        + "o atributo versionInfo est nulo.\n"
        + "Use setAlgorithmVersionInfo ou o construtor que recebe "
        + "algorithmVersionInfo.");
    }

    final FileInfo releaseNotes = versionInfo.getReleaseNotes();
    showReleaseNotesDialog(releaseNotes.getPath());
  }

  /**
   * Exibe o dilogo de ajuda com contedo PDF.
   *
   * @param simplePath Caminho para o contedo que deve ser exibido.
   */
  private void showReleaseNotesDialog(final String simplePath) {
    InputStream fileInputStream = null;
    try {
      fileInputStream = getStreamForReleaseNotes(versionInfo.getReleaseNotes()
        .getPath());
      String title = LNG.get("AlgorithmReleaseNotesAction.dialog.title",
        algorithmInfo.getName(), versionInfo.toString());
      final FileViewerDialog existingDialog = new FileViewerDialog(window,
        title, fileInputStream);
      existingDialog.setVisible(true);
      existingDialog.toFront();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Oculta o cone.
   */
  public void hideIcon() {
    putValue(Action.SMALL_ICON, null);
  }

  /**
   * Oculta o nome.
   */
  public void hideName() {
    putValue(Action.NAME, null);
  }

  /**
   * Associa Um algoritmo e uma verso de algoritmo a esta ao.
   *
   * @param versionInfo A verso do algoritmo.
   * @param algorithmInfo O algoritmo.
   */
  public void setAlgorithmVersionInfo(AlgorithmVersionInfo versionInfo,
    AlgorithmInfo algorithmInfo) {
    this.versionInfo = versionInfo;
    this.algorithmInfo = algorithmInfo;
    if (this.versionInfo == null) {
      setEnabled(false);
      return;
    }

    final FileInfo releaseNotes = this.versionInfo.getReleaseNotes();
    if (!containsReleaseNotes(releaseNotes)) {
      setEnabled(false);
    }
    else {
      setEnabled(true);
    }

  }

  /**
   * Verifica se existe um arquivo de release notes.
   *
   * @param releaseNotes arquivo de release notes.
   * @return indicativo.
   */
  private boolean containsReleaseNotes(FileInfo releaseNotes) {
    return releaseNotes != null && isResourceReleaseNotes(releaseNotes
      .getName());
  }

  /**
   * Verifica se o arquivo  um release notes.
   * 
   * @param filePath caminho para o arquivo.
   * @return Indicativo.
   */
  private boolean isResourceReleaseNotes(final String filePath) {
    return filePath.endsWith("releasenotes.txt");
  }

  /**
   * Obtm um input stream para o arquivo de release notes, a partir de seu
   * caminho.
   * 
   * @param filePath caminho do arquivo de release notes.
   * @return input stream do arquivo.
   * @throws Exception Erro no acesso ao arquivo.
   */
  private RemoteFileInputStream getStreamForReleaseNotes(final String filePath)
    throws Exception {
    final AlgorithmServiceInterface service =
      ClientRemoteLocator.algorithmService;
    final Task<RemoteFileInputStream> task = new Task<RemoteFileInputStream>() {
      @Override
      protected void performTask() throws Exception {
        final RemoteFileChannelInfo channel = service.prepareDownloadNotesFile(
          versionInfo.getInfo().getId(), versionInfo.getId(), filePath);
        final RemoteFileInputStream stream = new RemoteFileInputStream(channel);
        setResult(stream);
      }
    };

    task.execute(this.getWindow(), null, filePath);
    final Exception error = task.getError();
    if (error != null) {
      throw error;
    }

    if (task.wasCancelled()) {
      return null;
    }

    return task.getResult();
  }

  /**
   * Mostra o cone.
   */
  public void showIcon() {
    putValue(Action.SMALL_ICON, ApplicationImages.ICON_NOTES_24);
  }

  /**
   * Mostra o nome.
   */
  public void showName() {
    putValue(Action.NAME, LNG.get("algorithms.notes"));
  }

  /**
   * @return O owner desta ao.
   */
  private Window getWindow() {
    return window;
  }

}
