package csbase.client.util.csvpanel;

import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.JTableHeader;

import csbase.client.util.csvpanel.menu.ColumnGeneratorPopupMenu;
import csbase.client.util.csvpanel.menu.ColumnHeaderPopupMenu;
import csbase.client.util.csvpanel.menu.RowHeaderPopupMenu;
import csbase.client.util.csvpanel.table.CSVTable;
import csbase.client.util.csvpanel.table.CSVTableModel;
import csbase.client.util.event.EventListener;
import csbase.client.util.event.EventManager;
import csbase.exception.OperationFailureException;
import csbase.logic.ClientFile;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.table.RowHeaderTablePane;

/**
 * Painel a ser adicionado a uma aplicao, dando-lhe assim a funcionalidade de
 * visualizao de arquivos csv.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class CSVPanel extends JPanel {

  /**
   * Arquivo corrente que est sendo visualizado
   */
  private ClientFile file;

  /**
   * Gerente de eventos
   */
  private EventManager eventManager;

  /**
   * Tabela de dados.
   */
  private CSVTable table;

  /**
   * Painel com cabealho de linhas.
   */
  private RowHeaderTablePane rowHeaderScrollPane;

  /**
   * O menu popup do cabealho de coluna da tabela.
   */
  private ColumnHeaderPopupMenu headerPopup;

  /**
   * O menu popup do gerador de colunas da tabela.
   */
  private ColumnGeneratorPopupMenu cellPopup;

  /**
   * O menu popup do cabealho de linha da tabela.
   */
  private RowHeaderPopupMenu rowPopup;

  /**
   * Construtor.
   */
  public CSVPanel() {
    this.eventManager = new EventManager();
    createUI();
    this.headerPopup = new ColumnHeaderPopupMenu(table);
    this.cellPopup = new ColumnGeneratorPopupMenu(table);
    this.rowPopup = new RowHeaderPopupMenu(table);
  }

  /**
   * Cria os componentes visuais do painel.
   */
  private void createUI() {
    this.setLayout(new GridLayout());
    table = new CSVTable();
    table.getModel().addTableModelListener(new TableModelListener() {
      @Override
      public void tableChanged(TableModelEvent e) {
        // Ignora eventos que so s mudanas estruturais
        if (e.getFirstRow() != TableModelEvent.HEADER_ROW) {
          eventManager.fireEvent(new CSVTableModelNotification(e));
        }
        updateRowHeader();
      }
    });
    table.addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(MouseEvent e) {
        if (e.isPopupTrigger()) {
          cellPopup.show(e.getComponent(), e.getX(), e.getY());
        }
      }
    });
    rowHeaderScrollPane = new RowHeaderTablePane(table, Collections
      .emptyList());
    rowHeaderScrollPane.setCorner(ScrollPaneConstants.UPPER_LEFT_CORNER,
      new JButton());
    rowHeaderScrollPane.getRowHeader().getView().addMouseListener(
      new MouseAdapter() {
        @Override
        public void mousePressed(MouseEvent e) {
          if (e.isPopupTrigger()) {
            rowPopup.show(e.getComponent(), e.getX(), e.getY());
          }
        }
      });
    add(rowHeaderScrollPane);
    final JTableHeader header = table.getTableHeader();
    header.addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(MouseEvent e) {
        if (e.isPopupTrigger()) {
          headerPopup.show(e.getComponent(), e.getX(), e.getY());
        }
        else {
          int clickCount = e.getClickCount();
          if (clickCount == 1) {
            table.clearSelection();
          }
        }
      }
    });
    validate();
  }

  /**
   * Atualiza os cabealhos das linhas da tabela.
   */
  private void updateRowHeader() {
    int rowCount = table.getRowCount();
    List<Integer> nums = new ArrayList<Integer>();
    for (int i = 1; i <= rowCount; i++) {
      nums.add(i);
    }
    rowHeaderScrollPane.setRowNames(nums);
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link CSVExceptionNotification}.
   * 
   * @param listener o listener
   */
  public void addExceptionListener(
    EventListener<CSVExceptionNotification> listener) {
    eventManager.addEventListener(listener, CSVExceptionNotification.class);
  }

  /**
   * Armazena um {@link EventListener} para repassar a este eventos do tipo
   * {@link CSVExceptionNotification}.
   * 
   * @param listener o listener
   */
  public void addCSVTableModelListener(
    EventListener<CSVTableModelNotification> listener) {
    eventManager.addEventListener(listener, CSVTableModelNotification.class);
  }

  /**
   * Abre um arquivo CSV.
   * 
   * @param newFile o arquivo a ser aberto.
   * @param editable Indica se os dados podero ser editados na tabela.
   */
  public void openFile(ClientFile newFile, boolean editable) {
    /* Fechamos o arquivo corrente antes de abrir o novo arquivo. */
    closeFile();
    setFile(newFile, editable);
  }

  /**
   * Fecha o arquivo csv corrente.
   */
  public void closeFile() {
    if (getFile() != null) {
      setFile(null, false);
    }
  }

  /**
   * Obtm o arquivo CSV que est sendo visualizado.
   * 
   * @return o arquivo CSV ou nulo se no tiver nenhum arquivo aberto.
   */
  public ClientFile getFile() {
    return file;
  }

  /**
   * Atribui o arquivo como sendo o arquivo corrente e recarrega seus dados.
   * 
   * @param file o novo arquivo corrente.
   * @param editable Indica se os dados podero ser editados na tabela.
   */
  private void setFile(ClientFile file, boolean editable) {
    this.file = file;
    String[][] data = loadData(file);
    reloadTable(data, editable);
  }

  /**
   * Recarrega os dados da tabela.
   * 
   * @param data tabela de dados.
   * @param editable Indica se os dados podero ser editados na tabela.
   */
  private void reloadTable(String[][] data, boolean editable) {
    table.setData(data, editable);
  }

  /**
   * Atribui diretamente os dados a serem visualizados.
   * 
   * @param data a tabela de dados.
   * @param editable Indica se os dados podero ser editados na tabela.
   */
  public void setData(String[][] data, boolean editable) {
    closeFile();
    reloadTable(data, editable);
  }

  /**
   * Carrega os dados do arquivo corrente.
   * 
   * @param csvFile o arquivo CSV.
   * @return os dados lidos ou nulo em caso de erro.
   */
  private String[][] loadData(final ClientFile csvFile) {
    if (csvFile == null) {
      return null;
    }
    if (csvFile.size() <= 0) {
      return null;
    }
    try {
      return CSVFileUtils.readCSVFile(getOwner(), csvFile);
    }
    catch (OperationFailureException e) {
      eventManager.fireEvent(new CSVExceptionNotification(e, getString(
        "load.data.error")));
      return null;
    }
  }

  /**
   * Obtm a janela que detm o painel.
   * 
   * @return a janela que detm o painel.
   */
  public Window getOwner() {
    return SwingUtilities.windowForComponent(this);
  }

  /**
   * @return o ttulo da janela que detm esta instncia ou, caso este seja
   *         {@code null}, o retorno de
   *         {@link csbase.client.util.gui.log.LogPanel#getString(String, Object...)
   *         LogPanel#getString(String, Object...)} com o parmetro tag tendo o
   *         valor "title.default".
   */
  public String getTitle() {
    return getString("title");
  }

  /**
   * Obtm a string localizada da chave de internacionalizao especificada.
   * 
   * @param key a chave de internacionalizao.
   * @return a string localizada.
   */
  private String getString(String key) {
    return LNG.get(CSVPanel.class.getName() + "." + key);
  }

  /**
   * Indica se a tabela aceita edies.
   * 
   * @return verdadeiro se a tabela aceita edies ou falso, caso contrrio.
   */
  public boolean isEditable() {
    return table.isEditable();
  }

  /**
   * Salva o contedo da tabela CSV em um arquivo.
   * 
   * @param csvFile o arquivo.
   */
  public void saveFile(ClientFile csvFile) {
    try {
      String[][] data = getData();
      CSVFileUtils.writeCSVFile(getOwner(), data, csvFile);
      this.file = csvFile;
    }
    catch (OperationFailureException e) {
      eventManager.fireEvent(new CSVExceptionNotification(e, getString(
        "save.file.error")));
    }
  }

  /**
   * Obtm os dados atuais da tabela.
   * 
   * @return a tabela de dados.
   */
  private String[][] getData() {
    CSVTableModel tableModel = table.getModel();
    String[][] data = tableModel.getData();
    return data;
  }

  /**
   * Determina o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   * 
   * @param numRowsAsHeader nmero de linhas que devem ser consideradas como
   *        cabealho. Se igual a zero, no usa nenhuma das linhas no cabealho.
   */
  public void setNumRowsAsHeader(int numRowsAsHeader) {
    CSVTableModel tableModel = table.getModel();
    tableModel.setNumRowsAsHeader(numRowsAsHeader);
  }

  /**
   * Indica o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   * 
   * @return nmero de linhas que devem ser consideradas como cabealho. Se
   *         igual a zero, no usa nenhuma das linhas no cabealho.
   */
  public int getNumRowsAsHeader() {
    CSVTableModel tableModel = table.getModel();
    return tableModel.getNumRowsAsHeader();
  }

}
