package csbase.client.util.gui;

import java.awt.Component;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.ListIterator;

/**
 * A classe <code>CardsList</code> sincroniza a lista de referncias para os
 * componentes e a lista de nomes que referenciam os componentes. Cada par
 * [componente, nome] representa uma "carta" do CardPanel.
 */
class CardsList {

  /** Lista de componentes. */
  private LinkedList<Component> cards;

  /**
   * Lista de nomes associados aos componentes contidos na lista
   * <code>cards</code>.
   */
  private ArrayList<String> cardsNames;

  /** ndice da carta corrente. */
  private int currentCardindex;

  /**
   * Construtor.
   */
  CardsList() {
    cards = new LinkedList<Component>();
    cardsNames = new ArrayList<String>();
    currentCardindex = -1;
  }

  /**
   * Adiciona o componente  lista de componentes, associando-o ao nome recebido
   * como parmetro. Caso j exista referncia ao componente ou nome recebido
   * nas listas de deste onjeto, nada acontecer, e o mtodo retornar
   * <code>false</code>.
   * 
   * @param name Nome associado ao componente.
   * @param comp Referncia ao componente a ser adicionado.
   * @param index ndice da posio na lista de componentes onde este componente
   *        ser inserido. <code>-1</code> significa que o componente ser
   *        inserido ao final da lista.
   * @return True, caso o componente tenha sido adicionado com sucesso. False,
   *         caso contrrio.
   */
  boolean add(String name, Component comp, int index) {
    if (index > cards.size() || index < 0) {
      index = cards.size();
    }
    if (cardsNames.contains(name)) {
      throw new IllegalArgumentException("J existe card com este nome.");
    }
    if (cards.contains(comp)) {
      throw new IllegalArgumentException("J existe card com este componente.");
    }

    cardsNames.add(index, name);
    cards.add(index, comp);

    if (currentCardindex == -1) {
      currentCardindex = 0;
    }
    return true;
  }

  /**
   * Adiciona o componente  lista de componentes, associando-o ao nome recebido
   * como parmetro. Caso j exista referncia ao componente ou nome recebido
   * nas listas de deste onjeto, nada acontecer, e o mtodo retornar
   * <code>false</code>.
   * 
   * @param name Nome associado ao componente.
   * @param comp Referncia ao componente a ser adicionado.
   * @return True, caso o componente tenha sido adicionado com sucesso. False,
   *         caso contrrio.
   */
  boolean add(String name, Component comp) {
    return this.add(name, comp, -1);
  }

  /**
   * Adiciona o componente  lista de componentes, associando-o a um nome criado
   * arbitrariamente. Caso j exista referncia ao componente na lista de
   * "cartas", nada acontecer, e o mtodo retornar <code>null</code>.
   * 
   * @param comp Referncia ao componente a ser adicionado.
   * @return Nome criado para este componente, caso o componente tenha sido
   *         adicionado com sucesso, ou <code>null</code>, caso contrrio.
   */
  String add(Component comp) {
    return this.add(comp, -1);
  }

  /**
   * Adiciona o componente  lista de componentes, associando-o a um nome criado
   * arbitrariamente. Caso j exista referncia ao componente na lista de
   * "cartas", nada acontecer, e o mtodo retornar <code>null</code>.
   * 
   * @param comp Referncia ao componente a ser adicionado.
   * @param index ndice da posio na lista de componentes onde este componente
   *        ser inserido. <code>-1</code> significa que o componente ser
   *        inserido ao final da lista.
   * @return Nome criado para este componente, caso o componente tenha sido
   *         adicionado com sucesso, ou <code>null</code>, caso contrrio.
   */
  String add(Component comp, int index) {
    String name = comp.getClass().getSimpleName() + cards.size();
    if (add(name, comp, index)) {
      return name;
    }
    return null;
  }

  /**
   * Atualiza a carta corrente, mudando o valor de currentCardindex para o
   * ndice da carta que corresponde ao nome recebido como parmetro. Se o nome
   * recebido no corresponder a um componente adicionado na lista de
   * componentes desta classe, nada acontecer, e o mtodo retornar
   * <code>null</code>.
   * 
   * @param name Nome associado ao componente a ser exibido.
   * @return Referncia para o componente exibido, caso o nome recebido
   *         corresponda a um componente adicionado a lista de componentes desta
   *         classe, ou <code>null</code>, caso contrrio.
   */
  Component updateCurrentCard(String name) {
    if (!cardsNames.contains(name)) {
      return null;
    }
    int cardIndex = cardsNames.indexOf(name);
    currentCardindex = cardIndex;
    return cards.get(cardIndex);
  }

  /**
   * Retorna o nome associado ao componente recebido como parmetro. Se o nome
   * recebido no corresponder a um componente adicionado na lista de
   * componentes desta classe, o mtodo retornar <code>null</code>.
   * 
   * @param comp Referncia ao componente cujo nome associado est sendo
   *        consultado.
   * @return Nome associado a este componente, caso o componente esteja na lista
   *         de componentes desta classe, ou <code>null</code>, caso contrrio.
   */
  String getCardName(Component comp) {
    if (!cards.contains(comp)) {
      return null;
    }
    int cardIndex = cards.indexOf(comp);
    return cardsNames.get(cardIndex);
  }

  /**
   * Retorna o componente associado ao nome recebido como parmetro. Se o
   * componentew recebido no corresponder a um nome adicionado na lista de
   * nomes desta classe, o mtodo retornar <code>null</code>.
   * 
   * @param name Nome cujo nome componente associado est sendo consultado.
   * @return Componente associado a este nome, caso o nome esteja na lista de
   *         nomes desta classe, ou <code>null</code>, caso contrrio.
   */
  Component getComponentForCardName(String name) {
    if (!cardsNames.contains(name)) {
      return null;
    }
    int cardIndex = cardsNames.indexOf(name);
    return cards.get(cardIndex);
  }

  /**
   * @return O componente correspondente  "carta" que est em exibio, ou
   *         <code>null</code>, caso a lista de componentes esteja vazia.
   */
  Component getCurrentCard() {
    if (cards.isEmpty()) {
      return null;
    }
    return cards.get(currentCardindex);
  }

  /**
   * Remove o componente correspondente  este nome da lista de "cartas", e
   * remove este nome da lista de nomes.
   * 
   * @param name Nome do componente a ser removido.
   * @return True, caso houvesse componente associado ao nome recebido como
   *         parmetro. False, caso contrrio.
   */
  boolean remove(String name) {
    if (!cardsNames.contains(name)) {
      return false;
    }
    int cardIndex = cardsNames.indexOf(name);

    cardsNames.remove(cardIndex);
    cards.remove(cardIndex);

    correctCurrentCardIndex(cardIndex);
    return true;
  }

  /**
   * Remove o componente correspondente  referncia recebida como parmentro da
   * lista de "cartas", e remove o nome associado a este componente da lista de
   * nomes.
   * 
   * @param comp Referncia ao componente a ser removido.
   * @return True, caso a referncia recebida correspondesse a um elemento da
   *         lista de componentes desta classe. False, caso contrrio.
   */
  boolean remove(Component comp) {
    if (!cards.contains(comp)) {
      return false;
    }
    int cardIndex = cards.indexOf(comp);

    cardsNames.remove(cardIndex);
    cards.remove(cardIndex);

    correctCurrentCardIndex(cardIndex);
    return true;
  }

  /**
   * Atualiza o ndice de carta corrente (currentCardindex) e retorna o
   * componente seguinte da lista de "cartas". Se o componente corrente for o
   * ltimo da lista, o componente corrente passa a ser o primeiro componente da
   * lista.
   * 
   * @return O componente seguinte, ou <code>null</code>, caso as listas deste
   *         objeto (lista de cartas e de nomes) estejam vazia.
   */
  Component getNextCard() {
    if (cards.isEmpty()) {
      return null;
    }
    if (currentCardindex == (cards.size() - 1)) {
      /*
       * Se o elemento corrente for o ltimo da lista, o elemento corrente passa
       * a ser o primeiro elemento da lista.
       */
      currentCardindex = 0;
      return cards.getFirst();
    }
    ListIterator<Component> i = cards.listIterator(currentCardindex + 1);
    currentCardindex = i.nextIndex();
    return i.next();
  }

  /**
   * Atualiza o ndice de carta corrente (currentCardindex) e retorna o
   * componente anterior da lista de "cartas". Se o componente corrente for o
   * primeiro da lista, o componente corrente passa a ser o ltimo componente da
   * lista.
   * 
   * @return O componente anterior, ou <code>null</code>, caso as listas deste
   *         objeto (lista de cartas e de nomes) estejam vazia.
   */
  Component getPreviousCard() {
    if (cards.isEmpty()) {
      return null;
    }
    if (currentCardindex == 0) {
      /*
       * Se o elemento corrente for o primeiro da lista, o elemento corrente
       * passa a ser o ltimo elemento da lista.
       */
      currentCardindex = cards.size() - 1;
      return cards.getLast();
    }
    ListIterator<Component> i = cards.listIterator(currentCardindex);
    currentCardindex = i.previousIndex();
    return i.previous();
  }

  /**
   * Atualiza o ndice de carta corrente (currentCardindex) e retorna o primeiro
   * componente da lista de "cartas".
   * 
   * @return O primeiro componente da lista, ou <code>null</code>, caso as
   *         listas deste objeto (lista de cartas e de nomes) estejam vazia.
   */
  Component getFirstCard() {
    if (cards.isEmpty()) {
      return null;
    }
    currentCardindex = 0;
    return cards.getFirst();
  }

  /**
   * Atualiza o ndice de carta corrente (currentCardindex) e retorna o ltimo
   * componente da lista de "cartas".
   * 
   * @return O ltimo componente da lista, ou <code>null</code>, caso as listas
   *         deste objeto (lista de cartas e de nomes) estejam vazia.
   */
  Component getLastCard() {
    if (cards.isEmpty()) {
      return null;
    }
    currentCardindex = cards.size() - 1;
    return cards.getLast();
  }

  /**
   * Corrige o ndice da carta corrente aps a remoo de uma carta.
   * 
   * @param removedCardIndex
   */
  private void correctCurrentCardIndex(int removedCardIndex) {
    if (cards.isEmpty()) {
      /*
       * Se no existem mais cartas, o ndice da carta corrente passa a ser -1.
       */
      currentCardindex = -1;
      return;
    }
    if (currentCardindex > removedCardIndex) {
      /*
       * Se o ndice da carta corrente  maior do que o ndice da carta
       * removida, o ndice da carta corrente deve ser decrementado.
       */
      currentCardindex--;
      return;
    }
    if (currentCardindex == cards.size()) {
      /*
       * Se a carta corrente era a ltima carta da lista, o ndice da carta
       * corrente deve ser decrementado.
       */
      currentCardindex = 0;
      return;
    }

  }

}
