/*
 * $Id: SingleServerManager.java 149913 2014-02-18 19:26:51Z pietroguedes $
 */
package csbase.logic;

import java.rmi.RemoteException;

import csbase.exception.CSBaseException;

/**
 * <code>SingleServerManager</code>  a base para a monitorao de um nico
 * servidor.
 * 
 * @author Tecgraf/PUC-Rio
 */
public abstract class SingleServerManager extends ServerManager {

  /**
   * A instncia de monitorao
   */
  private SingleMonitoredServer singleMonitoredServer;

  /**
   * Implementa a monitorao do servidor.
   * 
   * @author Tecgraf/PUC-Rio
   * 
   */
  final class SingleMonitoredServer extends MonitoredServer {

    /**
     * Constri a monitorao
     * 
     * @param serverURI A URI do servidor a ser monitorado
     * @param locator A classe do locator
     * @param windowSize Valor mximo da janela de backoff.
     */
    protected SingleMonitoredServer(ServerURI serverURI, Class<?> locator,
      int windowSize) {
      this(serverURI, locator, true, windowSize);
    }

    /**
     * Constri a monitorao
     * 
     * @param serverURI A URI do servidor a ser monitorado
     * @param locator A classe do locator
     * @param verbose <code>true</code> se informaes devem ser exibidas,
     *        <code>false</code> se apenas erros e alertas devem ser exibidos
     * @param windowSize Valor mximo da janela de backoff.
     */
    protected SingleMonitoredServer(ServerURI serverURI, Class<?> locator,
      boolean verbose, int windowSize) {
      super(serverURI, locator, false, verbose, windowSize);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Session performLogin() throws CSBaseException, RemoteException {
      return SingleServerManager.this.login();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void postLogin() throws CSBaseException {
      /** nada a fazer */
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String lng(String key) {
      return lang(key);
    }

  }

  /**
   * Mtodo usado para traduo de termos
   * 
   * @param key A chave do termo
   * @return O valor traduzido
   */
  protected abstract String lang(String key);

  /**
   * Executa o login do servidor
   * 
   * @return A sesso do cliente logado
   * @throws CSBaseException em caso de falha no CSBase
   * @throws RemoteException em caso de erro de acesso remoto
   */
  protected abstract Session login() throws CSBaseException, RemoteException;

  /**
   * Constri o gerente
   * 
   * @param serverURI A URI do servidor a ser gerenciado
   * @param locator A classe do locator
   * @param windowSize Valor mximo da janela de backoff.
   */
  public SingleServerManager(ServerURI serverURI, Class<?> locator,
    int windowSize) {
    this(serverURI, locator, true, windowSize);
  }

  /**
   * Constri o gerente
   * 
   * @param serverURI A URI do servidor a ser gerenciado
   * @param locator A classe do locator
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   * @param windowSize Valor mximo da janela de backoff.
   */
  public SingleServerManager(ServerURI serverURI, Class<?> locator,
    boolean verbose, int windowSize) {
    super(true, verbose);
    if (serverURI == null) {
      throw new IllegalArgumentException("URI no pode ser nula!");
    }
    this.singleMonitoredServer =
      new SingleMonitoredServer(serverURI, locator, verbose, windowSize);
    this.addServer(this.singleMonitoredServer);
  }

  /**
   * Tenta estebelecer comunicao com o servidor. Em caso de sucesso, efetua o
   * login do usurio, obtm os servios e instala todos os observadores remotos
   * no servidor.
   * 
   * @return true em caso de sucesso, false caso contrrio
   * 
   * @throws CSBaseException Caso o servidor no esteja disponvel.
   * @throws RemoteException Em caso de falha na comunicao
   */
  public final boolean lookup() throws CSBaseException, RemoteException {
    if (super.performLookup()) {
      if (super.validate()) {
        return true;
      }
      return false;
    }
    throw new CSBaseException("Servidor fora do ar.");
  }

  /**
   * Interrompe o monitor. Remove os observadores, executa o logout
   * 
   */
  public final synchronized void stop() {
    this.getMonitor().stopMonitoring();
    this.removeObservers();
    this.logout(true);
  }

  /**
   * Inicia a monitorao do servidor
   */
  public final void startMonitoring() {
    this.getMonitor().startMonitoring();
  }

  /**
   * Testa se o servidor est alcanvel.
   * 
   * @return true, caso o servidor esteja alcanvel, ou false, caso contrrio.
   */
  public final boolean ping() {
    return this.getMonitor().ping();
  }

  /**
   * Define o nome do sistema a ser usado
   * 
   * @param systemName O nome do sistema
   */
  public final void setSystemName(String systemName) {
    this.setSystemName(this.singleMonitoredServer.getURI(), systemName);
  }

  /**
   * Executa o logout no servidor e o retira da monitorao
   */
  public final void logout() {
    this.logout(true);
  }
}
