package csbase.logic.algorithms.xml.category;

import java.io.IOException;
import java.io.Writer;
import java.util.List;

import tecgraf.javautils.xml.ImprovedXMLListIterator;
import tecgraf.javautils.xml.XMLBasicElement;
import tecgraf.javautils.xml.XMLComment;
import tecgraf.javautils.xml.XMLElementInterface;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;

/**
 * Esta classe implementa o elemento XML raiz do documento que descreve todas as
 * categorias de algoritmos definidas no sistema.
 * 
 */
public class XmlCategoriesElement extends XMLBasicElement {

  /**
   * Constri o elemento xml que descreve todas as categorias de algoritmos.
   */
  public XmlCategoriesElement() {
    super();
    setTag(XmlCategoriesUtils.CATEGORIES_TAG);
  }

  /**
   * Obtm o conjunto de categorias que representa o contexto da aplicao.
   * 
   * @return o conjunto de categoria de algoritmos
   */
  public CategorySet getCategorySet() {
    return (CategorySet) getAppContextObject();
  }

  /**
   * Executado quando a tag de fim do elemento XML  encontrada. Deve-se criar o
   * objeto correspondente na camada da aplicao.
   * 
   * @see tecgraf.javautils.xml.XMLElementInterface#endTag(java.util.List)
   */
  public void endTag(List<XMLElementInterface> childrenList) {
    ImprovedXMLListIterator iterator = new ImprovedXMLListIterator(childrenList);
    while (iterator.hasNext()) {
      XMLElementInterface nextElem = iterator.next();
      Category category = ((XmlCategoryElement) nextElem).getCategory();
      // System.out.println("  ----> Categoria adicionada ao conjunto: "
      // + category);
      if (category != null) {
        getCategorySet().addCategory(category);
      }
    }
  }

  /**
   * Escreve todos os elementos que descrevem as categorias de algoritmos.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @throws IOException
   */
  private void writeAllCategoryElement(Writer writer, String ident)
    throws IOException {
    String newIdent = XMLBasicElement.getNextIdentation(ident);

    for (Category category : getCategorySet().getCategories()) {
      writeCategoryElement(writer, newIdent, category);
    }
  }

  /**
   * Escreve o elemento que descreve uma categoria de algoritmos.
   * 
   * @param writer stream de sada
   * @param ident identao corrente
   * @param category categoria de algoritmo a ser escrita
   * @throws IOException
   */
  private void writeCategoryElement(final Writer writer, final String ident,
    Category category) throws IOException {
    XmlCategoryElement xmlcategory = new XmlCategoryElement(category);
    xmlcategory.write(writer, ident);
  }

  @Override
  public void write(Writer writer, String ident) throws IOException {
    writeStartTagln(writer, ident);
    String comment = "Especificando as categorias de algoritmos";
    XMLComment.write(writer, ident, comment);
    writeAllCategoryElement(writer, ident);
    writeEndTag(writer, ident);
  }
}
