/*
 * IdFactory.java
 * 
 * $Author: pizzol $ $Revision: 105329 $ - $Date: 2006-08-24 10:23:50 -0300 (Thu,
 * 24 Aug 2006) $
 */
package csbase.logic;

import java.io.Serializable;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;

/**
 * A classe IdFactory  responsvel pela gerao dos identificadores numricos.
 * Estas classes devem implementar a interface logic.IdInterface. O
 * identificador  um inteiro sequencial. O algoritmo reutiliza os valores de
 * identificadores liberados pela remoo.
 */
public class IdFactory implements Serializable {
  private int nextId;
  private List<Object> freeId;

  /**
   * Constri um gerador de identificadores de IdInterfaces
   * 
   * @param idInterfaces .
   */
  public IdFactory(List<? extends IdInterface> idInterfaces) {
    Collections.sort(idInterfaces, getComparator());
    freeId = new LinkedList<Object>();
    if (idInterfaces.size() == 0) {
      nextId = 0;
    }
    else {
      // Pega os ids livres at o menor id existente.
      IdInterface idInterface = idInterfaces.get(0);
      int smallerId = ((Integer) idInterface.getId()).intValue();
      for (int t = 0; t < smallerId; t++) {
        freeId.add(new Integer(t));
      }
      nextId = smallerId + 1;
      // Loop ate o penultimo elemento
      for (int i = 0; i < (idInterfaces.size() - 1); i++) {
        idInterface = (IdInterface) idInterfaces.get(i);
        int id = ((Integer) idInterface.getId()).intValue();
        IdInterface idInterfaceNext = (IdInterface) idInterfaces.get(i + 1);
        int idNext = ((Integer) idInterfaceNext.getId()).intValue();
        // verifica se tem um numero vago entre
        // dois ids
        for (int j = id; j < (idNext - 1); j++) {
          freeId.add(new Integer(j + 1));
        }
        nextId = idNext + 1;
      }
    }
  }

  /**
   * Obtm o prximo identificador disponvel.
   * 
   * @return o prximo identificador disponvel
   */
  public Object next() {
    if (freeId.size() == 0) {
      int current = nextId;
      nextId++;
      return new Integer(current);
    }
    return freeId.remove(0);
  }

  /**
   * Libera um identificador para ser reutilizado.
   * 
   * @param id o identificador liberado
   */
  public void free(Object id) {
    freeId.add(id);
  }

  /**
   * Obtm um comparator de <code>IdInterface</code> pelo critrio de ordem dos
   * identificadores.
   * 
   * @return um comparador por identificador
   */
  private Comparator<IdInterface> getComparator() {
    Comparator<IdInterface> idInterfaceComparator = new Comparator<IdInterface>() {
      public int compare(IdInterface idInterface1, IdInterface idInterface2) {
        Integer id1 = (Integer) idInterface1.getId();
        Integer id2 = (Integer) idInterface2.getId();
        return id1.compareTo(id2);
      }
    };
    return idInterfaceComparator;
  }
}
