/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;

import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;
import csbase.logic.algorithms.parameters.validators.TextParameterValidator;

/**
 * <p>
 * Coluna de texto
 * </p>
 *
 * <p>
 * Uma coluna para o parmetro do tipo tabela {@link TableParameter} que aceita
 * textos.
 * </p>
 *
 * @author lmoreira
 */
public final class TextColumn extends BasicColumn<String> {

  /**
   * O nmero mximo de caracteres.
   */
  private Integer maxCharacters;

  /**
   * Cria uma coluna.
   *
   * @param id O identificador da coluna (No aceita {@code null}).
   * @param label O rtulo da coluna (No aceita {@code null}).
   * @param defaultValue O valor-padro da coluna (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional ({@code true})
   *        ou {@code false}.
   * @param isEditable Indica se a coluna  editvel.
   * @param maxCharacters O nmero mximo de caracteres para que o valor de uma
   *        clula seja considerado vlido (Aceita {@code null}).
   */
  public TextColumn(String id, String label, String defaultValue,
    boolean isOptional, boolean isEditable, Integer maxCharacters) {
    super(id, label, defaultValue, isOptional, isEditable);
    setMaxCharacters(maxCharacters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getItemValueFromText(String itemValue) {
    return itemValue;
  }

  /**
   * Obtm o nmero mximo de caracteres para que o valor de uma clula seja
   * considerado vlido.
   *
   * @return O nmero mximo de caracteres ou {@code null} caso no haja um
   *         nmero mximo de caracteres definido.
   */
  public Integer getMaxCharacters() {
    return this.maxCharacters;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SimpleParameterValidator<String> getValidator() {
    return new TextParameterValidator(isOptional(), getMaxCharacters());
  }

  /**
   * Atribui o nmero mximo de caracteres para que o valor de uma clula seja
   * considerado vlido.
   *
   * @param maxCharacters O nmero mximo de caracteres ou {@code null} caso no
   *        haja um nmero mximo de caracteres definido.
   */
  private void setMaxCharacters(Integer maxCharacters) {
    if (maxCharacters != null && maxCharacters.intValue() <= 0) {
      throw new IllegalArgumentException(MessageFormat.format(
        "O parmetro maxCharacters deveria ser positivo.\n"
          + "Valor encontrado: {0}.", new Object[] { maxCharacters }));
    }
    this.maxCharacters = maxCharacters;
  }
}
