/*
 * CommandEndNotificationData.java
 * 
 * $Author: isabella $ $Date: 2016-07-07 14:52:00 -0300 (Thu, 07 Jul 2016) $
 * $Revision: 174851 $
 */

package csbase.logic;

/**
 * Classe que representa os dados de mensagem de fim de comando em um SGA,
 * quando o comando no d informaes sobre como a execuo terminou (com
 * sucesso ou com erro).
 */
public class CommandEndNotification extends
  ExtendedCommandFinishedNotification {

  /**
   * Indica se os tempos de execuo dos comandos esto definidos na notificao
   */
  private boolean timeInfo;

  /**
   * Retorna verdadeiro se possui os tempos de execuo dos comandos.
   *
   * @return verdadeiro se possui os tempos de execuo dos comandos.
   */
  public boolean hasTimeInfo() {
    return timeInfo;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getStatusString() {
    return "END";
  }

  /**
   * Constri o evento de trmino normal de comando.
   *
   * @param sender Quem est enviando a notificao (no pode ser {@code null}).
   * @param commandId identificador do comando.
   * @param cmdDesc descrio do comando.
   * @param tip Dica para auxiliar a identificao do algoritmo do comando.
   * @param elapsedTimeSec tempo de parede.
   * @param userTimeSec tempo em modo usurio.
   * @param cpuTimeSec tempo em modo sistema.
   * @param startTime Hora de incio
   * @param endTime Hora de trmino.
   * @param execHost Servidor de execuo.
   * @param projectId identificador do projeto
   * @param finalizationInfo Informaes sobre a finalizao do comando.
   */
  public CommandEndNotification(String sender, Object commandId,
    String cmdDesc, String tip, Integer elapsedTimeSec, Integer userTimeSec,
    Integer cpuTimeSec, long startTime, long endTime, String execHost,
    Object projectId, CommandFinalizationInfo finalizationInfo) {
    super(sender, commandId, cmdDesc, tip, elapsedTimeSec, userTimeSec,
      cpuTimeSec, startTime, endTime, execHost, projectId, finalizationInfo);
    this.timeInfo =
      (elapsedTimeSec != null && elapsedTimeSec >= 0)
        || (userTimeSec != null && userTimeSec >= 0)
        || (cpuTimeSec != null && cpuTimeSec >= 0);
  }

  /**
   * Constri o evento de trmino de comando sem dados de tempo.
   *
   * @param sender Quem est enviando a notificao (no pode ser {@code null}).
   * @param commandId identificador do comando.
   * @param cmdDesc descrio do comando.
   * @param tip Dica para auxiliar a identificao do algoritmo do comando.
   * @param startTime Hora de incio
   * @param endTime Hora de trmino.
   * @param execHost Servidor de execuo.
   * @param projectId identificador do projeto.
   * @param finalizationInfo Informaes sobre a finalizao do comando.
   */
  public CommandEndNotification(String sender, Object commandId,
    String cmdDesc, String tip, long startTime, long endTime, String execHost,
    Object projectId, CommandFinalizationInfo finalizationInfo) {
    super(sender, commandId, cmdDesc, tip, null, null, null, startTime,
      endTime, execHost, projectId, finalizationInfo);
    this.timeInfo = false;
  }

  /**
   * Constri o evento de trmino de comando sem dados de tempo.
   *
   * @param sender Quem est enviando a notificao (no pode ser {@code null}).
   * @param commandId identificador do comando.
   * @param cmdDesc descrio do comando.
   * @param tip Dica para auxiliar a identificao do algoritmo do comando.
   * @param startTime Hora de incio
   * @param endTime Hora de trmino.
   * @param execHost Servidor de execuo.
   * @param projectId identificador do projeto.
   */
  public CommandEndNotification(String sender, Object commandId, String cmdDesc,
    String tip, long startTime, long endTime, String execHost,
    boolean hasWarnings, Object projectId) {
    this(sender, commandId, cmdDesc, tip, startTime, endTime, execHost,
      projectId, new SimpleCommandFinalizationInfo(CommandFinalizationType.END,
        hasWarnings));
  }
}
