/*
 * ExtendedCommandFinishedNotificationData.java
 * 
 * $Author: taciana $ $Date: 2007-06-26 12:18:26 -0300 (Tue, 26 Jun 2007) $
 * $Revision: 64808 $
 */

package csbase.logic;

/**
 * Classe que representa os dados mais detalhados de mensagem de fim de comando
 * em um SGA. Essa classe deve ser estendida pela notificaes de comando que
 * contm esses dados mais detalhados.
 */
public abstract class ExtendedCommandFinishedNotification extends
  CommandNotification {

  /** Tempo de parede do comando. */
  protected Integer elapsedTimeSec;

  /** Tempo de usurio do comando. */
  protected Integer userTimeSec;

  /** Tempo de CPU do comando. */
  protected Integer cpuTimeSec;

  /** hora de trmino */
  protected long endTime;

  /**
   * Constri o evento de trmino de comando.
   * 
   * @param sender Quem est enviando a notificao (no pode ser {@code null}).
   * @param commandId identificador do comando.
   * @param cmdDesc descrio do comando.
   * @param tip Dica para auxiliar a identificao do algoritmo do comando.
   * @param elapsedTimeSec tempo de parede.
   * @param userTimeSec tempo em modo usurio.
   * @param cpuTimeSec tempo em modo sistema.
   * @param startTime Hora de incio
   * @param endTime Hora de trmino.
   * @param execHost Servidor de execuo.
   * @param projectId identificador do projeto
   * @param finalizationInfo Informaes sobre a finalizao do comando.
   */
  public ExtendedCommandFinishedNotification(String sender,
    Object commandId, String cmdDesc, String tip, Integer elapsedTimeSec,
    Integer userTimeSec, Integer cpuTimeSec, long startTime, long endTime,
    String execHost, Object projectId, CommandFinalizationInfo finalizationInfo) {
    super(sender, commandId, cmdDesc, tip, startTime, execHost, projectId,
      finalizationInfo);
    this.endTime = endTime;
    setElapsedTime(elapsedTimeSec);
    setUserTime(userTimeSec);
    setCpuTime(cpuTimeSec);
  }

  /**
   * String para auditagem.
   * 
   * @return uma string
   */
  @Override
  public String toString() {
    String msg =
      "CMD/" + getStatusString() + " (" + cmdDesc + ") (" + getTip() + ")"
        + " - " + commandId + " : ";
    if (finalizationInfo.getExitCode() != null) {
      msg = msg + "cdigo de sada " + finalizationInfo.getExitCode() + ", ";
    }
    else {
      msg = msg + "cdigo de sada <no informado>, ";
    }
    if (userTimeSec != null) {
      msg = msg + "user " + userTimeSec + " s, ";
    }
    else {
      msg = msg + "user <no informado>, ";
    }
    if (cpuTimeSec != null) {
      msg = msg + "cpu " + cpuTimeSec + " s, ";
    }
    else {
      msg = msg + "cpu <no informado>, ";
    }
    if (elapsedTimeSec != null) {
      msg = msg + "total " + elapsedTimeSec + " s.";
    }
    else {
      msg = msg + "total <no informado>.";
    }
    msg = msg + "\n";
    if (startTime >= 0) {
      msg = msg + "hora de incio " + startTime + ".";
    }
    else {
      msg = msg + "hora de incio <no informado>.";
    }
    if (endTime >= 0) {
      msg = msg + "hora de trmino " + endTime + ".";
    }
    else {
      msg = msg + "hora de trmino <no informado>.";
    }
    if ((execHost != null) && (!execHost.equals(""))) {
      msg = msg + "executado por " + execHost + ".";
    }
    else {
      msg = msg + "executado por <no informado>.";
    }
    return msg;
  }

  /**
   * Retorna o tempo de parede do comando.
   * 
   * @return o tempo
   */
  public Integer getElapsedTime() {
    return elapsedTimeSec;
  }

  /**
   * Retorna o tempo em modo usurio.
   * 
   * @return o tempo
   */
  public Integer getUserTime() {
    return userTimeSec;
  }

  /**
   * Retorna o tempo de sistema.
   * 
   * @return o tempo
   */
  public Integer getCPUTime() {
    return cpuTimeSec;
  }

  /**
   * Retorna a hora de trmino da execuo do comando.
   * 
   * @return o identificador
   */
  public long getEndTime() {
    return endTime;
  }

  /**
   * Atribui o tempo de parede.
   * 
   * @param elapsedTime O tempo de parede.
   */
  private void setElapsedTime(Integer elapsedTime) {
    if (elapsedTime != null && elapsedTime < 0) {
      throw new IllegalArgumentException("elapsedTime < 0 ");
    }

    this.elapsedTimeSec = elapsedTime;
  }

  /**
   * Atribui o tempo do usurio.
   * 
   * @param userTime O tempo do usurio.
   */
  private void setUserTime(Integer userTime) {
    if (userTime != null && userTime < 0) {
      throw new IllegalArgumentException("userTime < 0 ");
    }

    this.userTimeSec = userTime;
  }

  /**
   * Atribui o tempo do sistema.
   * 
   * @param cpuTime O tempo do sistema.
   */
  private void setCpuTime(Integer cpuTime) {
    if (cpuTime != null && cpuTime < 0) {
      throw new IllegalArgumentException("cpuTime < 0 ");
    }

    this.cpuTimeSec = cpuTime;
  }
}
