package csbase.logic;

import tecgraf.javautils.version.VersionNumber;

/**
 * Verso de uma classe (lida do manifest do jar).
 *
 * @author Tecgraf
 */
public class ManifestVersion {

  /**
   * String associada a uma verso no lanada.
   */
  public static final String BAD_VERSION_NAME = "??.??.??";

  /**
   * Nmero da verso.
   */
  private VersionNumber versionNumber;

  /**
   * Construtor.
   *
   * @param clazz a classe a partir da qual a verso ser lida.
   */
  public ManifestVersion(Class<?> clazz) {
    if (clazz == null) {
      throw new IllegalArgumentException("Argument clazz cannot be nul.");
    }
    versionNumber = readVersion(clazz);
  }

  /**
   * L o nmero da verso da classe a partir dos atributos do Manifest.
   *
   * @param clazz a classe
   * @return o nmero da verso.
   */
  protected VersionNumber readVersion(Class<?> clazz) {
    String versionString = clazz.getPackage().getImplementationVersion();
    if (versionString != null) {
      return VersionNumber.fromString(versionString);
    }
    return null;
  }

  /**
   * Obtm o nome da verso.
   *
   * @return O nome da verso.
   */
  final public String getVersionName() {
    return versionNumber == null ? BAD_VERSION_NAME : versionNumber.toString();
  }

  /**
   * Retorna o nmero da <i>major version</i>.
   *
   * @return nmero da major version, ou -1 caso a verso no tenha sido
   *         calculada
   */
  final public int getMajorVersion() {
    return versionNumber == null ? -1 : versionNumber.getMajor();
  }

  /**
   * Retorna o nmero da <i>minor version</i>.
   *
   * @return nmero da patch version, ou -1 caso a verso no tenha sido
   *         calculada
   */
  final public int getMinorVersion() {
    return versionNumber == null ? -1 : versionNumber.getMinor();
  }

  /**
   * Retorna o nmero da <i>patch version</i>.
   *
   * @return nmero da patch version, ou -1 caso a verso no tenha sido
   *         calculada
   */
  final public int getPatchVersion() {
    return versionNumber == null ? -1 : versionNumber.getPatch();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return getVersionName();
  }
}
