package csbase.logic.algorithms.parameters;

import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.BooleanParameterValidator;
import csbase.logic.algorithms.parameters.validators.SimpleParameterValidator;

/**
 * <p>
 * Coluna de booleanos.
 * </p>
 *
 * <p>
 * A coluna de booleanos  de preenchimento obrigatrio.
 * </p>
 *
 * @author lmoreira
 */
public final class BooleanColumn extends BasicColumn<Boolean> {
  /**
   * O valor-padro para o comando quando o valor da clula  {@code true}.
   */
  private static final String DEFAULT_TRUE_COMMAND = "1";

  /**
   * O valor-padro para o comando quando o valor da clula  {@code false}.
   */
  private static final String DEFAULT_FALSE_COMMAND = "0";

  /**
   * O valor do comando quando o valor da clula  {@code false}.
   */
  private String falseCommand;

  /**
   * O valor do comando quando o valor da clula  {@code true}.
   */
  private String trueCommand;

  /**
   * Cria uma coluna de booleanos.
   *
   * @param id O identificador (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param defaultValue O valor-padro.
   * @param isEditable Indica se a coluna  editvel.
   * @param falseCommand O valor para o comando quando o valor da clula for
   *        {@code false}. Se for {@code null} o valor do comando ser
   *        {@value #DEFAULT_FALSE_COMMAND}.
   * @param trueCommand O valor para o comando quando o valor da clula for
   *        {@code true}. Se for {@code null} o valor do comando ser
   *        {@value #DEFAULT_TRUE_COMMAND}.
   */
  public BooleanColumn(String id, String label, boolean defaultValue,
    boolean isEditable, String falseCommand, String trueCommand) {
    super(id, label, defaultValue, false, isEditable);
    setFalseCommand(falseCommand);
    setTrueCommand(trueCommand);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandValue(Boolean itemValue, CommandLineContext context) {
    if (itemValue == null) {
      throw new IllegalArgumentException(
        "O parmetro itemValue no pode ser nulo.");
    }
    if (context == null) {
      throw new IllegalArgumentException(
        "O parmetro context no pode ser nulo.");
    }
    if (itemValue) {
      return getTrueCommand();
    }
    return getFalseCommand();
  }

  /**
   * Obtm o valor da linha de comando quando o valor da clula  {@code false}.
   *
   * @return .
   */
  public String getFalseCommand() {
    return this.falseCommand;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Boolean getItemValueFromText(String itemValue) throws ParseException {
    if (itemValue == null) {
      throw new ParseException(
        "No  possvel atribuir um valor nulo a uma coluna de booleanos.\n"
          + "Coluna envolvida: {0}.", new Object[] { this });
    }
    if (Boolean.TRUE.toString().equals(itemValue)) {
      return new Boolean(true);
    }
    if (Boolean.FALSE.toString().equals(itemValue)) {
      return new Boolean(false);
    }
    throw new ParseException("O valor {0} foi atribuido  coluna {1}, "
      + "porm ele no  um valor vlido para um booleano.", new Object[] {
          itemValue, this });
  }

  /**
   * Obtm o valor da linha de comando quando o valor da clula  {@code true}.
   *
   * @return .
   */
  public String getTrueCommand() {
    return this.trueCommand;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SimpleParameterValidator<Boolean> getValidator() {
    return new BooleanParameterValidator();
  }

  /**
   * <p>
   * Atribui o valor da linha de comando quando o valor do parmetro  falso.
   * </p>
   *
   * <p>
   * Se o parmetro {@code falseCommand} for {@code null}, este valor ser igual
   * a {@link #DEFAULT_FALSE_COMMAND} ({@value #DEFAULT_FALSE_COMMAND})
   * </p>
   *
   * @param command O valor (Aceita {@code null}).
   */
  private void setFalseCommand(String command) {
    String cmd = command;
    if (cmd == null) {
      cmd = DEFAULT_FALSE_COMMAND;
    }
    this.falseCommand = cmd;
  }

  /**
   * <p>
   * Atribui o valor da linha de comando quando o valor do parmetro 
   * verdadeiro.
   * </p>
   *
   * <p>
   * Se o parmetro {@code trueCommand} for {@code null}, este valor ser igual
   * a {@link #DEFAULT_TRUE_COMMAND} ({@value #DEFAULT_TRUE_COMMAND})
   * </p>
   *
   * @param command O valor (Aceita {@code null}).
   */
  private void setTrueCommand(String command) {
    String cmd = command;
    if (cmd == null) {
      cmd = DEFAULT_TRUE_COMMAND;
    }
    this.trueCommand = cmd;
  }
}
