/*
 * $Author: lmoreira $ $Date: 2005/07/14 14:28:19 $ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;

import tecgraf.openbus.algorithmservice.v1_0.parameters.DoubleParameterHelper;
import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.validators.DoubleParameterValidator;

/**
 * Parmetro do tipo real.
 *
 * @author lmoreira
 */
public final class DoubleParameter extends SimpleParameter<Double> {

  /**
   * Valor indicativo de erro.
   */
  public static final double ERROR_VALUE = Double.NaN;

  /**
   * Indica se o valor mximo est entre os valores vlidos.
   */
  private boolean isMaximumIncluded;
  /**
   * Indica se o valor mnimo est entre os valores vlidos.
   */
  private boolean isMinimumIncluded;
  /**
   * O valor mximo.
   */
  private Double maximum;
  /**
   * O valor mnimo.
   */
  private Double minimum;

  /**
   * Os observadores.
   */
  private transient List<DoubleParameterListener> listeners;

  /**
   * Cria um parmetro.
   *
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor  opcional/obrigatrio.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param maximum O mximo (Aceita {@code null}).
   * @param isMaximumIncluded Indica se o valor mximo est entre os valores
   *        vlidos.
   * @param minimum O mnimo (Aceita {@code null}).
   * @param isMinimumIncluded Indica se o valor mnimo est entre os valores
   *        vlidos.
   */
  public DoubleParameter(String name, String label, String description,
    Double defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, Double maximum, boolean isMaximumIncluded,
    Double minimum, boolean isMinimumIncluded) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    this.listeners = new LinkedList<DoubleParameterListener>();
    setMaximum(maximum, isMaximumIncluded);
    setMinimum(minimum, isMinimumIncluded);
  }

  /**
   * Adiciona o observador a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public void addDoubleParameterListener(DoubleParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return getValue();
  }

  /**
   * Obtm o mximo.
   *
   * @return O mximo ou {@code null} se ele no existir.
   */
  public Double getMaximum() {
    return this.maximum;
  }

  /**
   * Obtm o mnimo.
   *
   * @return O mnimo ou {@code null} se ele no existir.
   */
  public Double getMinimum() {
    return this.minimum;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return "DOUBLE";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return DoubleParameterHelper.id();
  }

  /**
   * Indica se o valor mximo est entre os valores vlidos.
   *
   * @return .
   */
  public boolean isMaximumIncluded() {
    return this.isMaximumIncluded;
  }

  /**
   * Indica se o valor mnimo est entre os valores vlidos.
   *
   * @return .
   */
  public boolean isMinimumIncluded() {
    return this.isMinimumIncluded;
  }

  /**
   * <p>
   * Atribui o mximo e se o mximo  includo.
   * </p>
   *
   * <p>
   * Gera o evento
   * {@link DoubleParameterListener#minimumWasChanged(DoubleParameter)}.
   * </p>
   *
   * @param maximum O mximo (Aceita {@code null}).
   * @param maximumIncluded Indica se o mximo deve ou no ser considerado um
   *        valor vlido.
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o mximo
   *         informado  igual ao mximo anterior.
   */
  public boolean setMaximum(Double maximum, boolean maximumIncluded) {
    if (minimum != null) {
      if ((maximum != null) && (maximum.compareTo(minimum) < 0)) {
        throw new IllegalArgumentException(
          "O valor mximo no pode ser menor do que o valor mnimo.");
      }
    }
    boolean maxIncluded = maximumIncluded;
    if (maximum == null) {
      if (this.maximum == null) {
        return false;
      }
      maxIncluded = false;
    }
    else if (maximum.equals(this.maximum)) {
      return false;
    }

    this.isMaximumIncluded = maxIncluded;
    this.maximum = maximum;
    updateValidator();
    for (DoubleParameterListener listener : listeners) {
      listener.maximumWasChanged(this);
    }
    return true;
  }

  /**
   * <p>
   * Atribui o mnimo e se o minimo  includo.
   * </p>
   *
   * <p>
   * Gera o evento
   * {@link DoubleParameterListener#minimumWasChanged(DoubleParameter)}.
   * </p>
   *
   * @param minimum O mnimo (Aceita {@code null}).
   * @param minimumIncluded Indica se o mnimo deve ou no ser considerado um
   *        valor vlido.
   *
   * @return {@code true} em caso de sucesso ou {@code false} se o mnimo
   *         informado  igual ao mpinimo anterior.
   */
  public boolean setMinimum(Double minimum, boolean minimumIncluded) {
    if (maximum != null) {
      if ((minimum != null) && (minimum.compareTo(maximum) > 0)) {
        throw new IllegalArgumentException(
          "O valor mnimo no pode ser maior do que o valor mximo.");
      }
    }
    boolean minIncluded = minimumIncluded;
    if (minimum == null) {
      if (this.minimum == null) {
        return false;
      }
      minIncluded = false;
    }
    else if (minimum.equals(this.minimum)) {
      return false;
    }
    this.isMinimumIncluded = minIncluded;
    this.minimum = minimum;
    updateValidator();
    for (DoubleParameterListener listener : listeners) {
      listener.minimumWasChanged(this);
    }
    return true;
  }

  /**
   * Atualiza os valores utilizados para validar o parmetro.
   */
  private void updateValidator() {
    DoubleParameterValidator validator =
      (DoubleParameterValidator) getParameterValidator();
    validator.setMaximum(this.maximum, this.isMaximumIncluded);
    validator.setMinimum(this.minimum, this.isMinimumIncluded);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if (parameterValue != null) {
      Double value;
      try {
        value = new Double(Double.parseDouble(parameterValue));
      }
      catch (NumberFormatException e) {
        throw new ParseException(e,
          "No  possvel atribuir o valor {0} a um parmetro do tipo real.\n"
            + "Parmetro envolvido: {1}.\n", parameterValue, this);
      }
      setValue(value);
    }
    else {
      setValue(null);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DoubleParameterValidator createParameterValidator() {
    return new DoubleParameterValidator(isOptional(), maximum,
      isMaximumIncluded, minimum, isMinimumIncluded);
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in Leitor de objetos
   *
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
  ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<DoubleParameterListener>();
  }
}
