package csbase.logic.algorithms.parameters;

import java.util.HashMap;
import java.util.Map;

import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.CommandLineParameterFormatter;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * <p>
 * Parmetro oculto.
 * </p>
 * 
 * <p>
 * Um parmetro oculto no aparece na interface grfica do configurador de
 * algoritmos, mas aparece na linha de comando. Ele  utilizado para passar ao
 * comando parmetros que no variam.
 * </p>
 * 
 * <p>
 * Este parmetro  somente-leitura. Todos os mtodos de parmetros que
 * modificam o valor do comando no tm efeito.
 * </p>
 * 
 * @author lmoreira
 */
public final class HiddenParameter extends Parameter<String> {

  /**
   * O formato no comando.
   */
  private String commandLinePattern;

  /**
   * O valor do parmetro.
   */
  private String value;

  /**
   * Cria um parmetro oculto.
   * 
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param value O valor do parmetro (No aceita {@code null}).
   * @param commandLinePattern O padro para a linha de comando (No aceita
   *        {@code null}).
   */
  public HiddenParameter(String name, String value, String commandLinePattern) {
    super(name);
    setValue(value);
    setCommandLinePattern(commandLinePattern);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    return new HashMap<String, Object>();
  }

  /**
   * <p>
   * Obtm a linha de comando.
   * </p>
   * 
   * <p>
   * A linha de comando tem o formato
   * <b>Nome_do_parametro=Valor_do_parametro</b>.
   * </p>
   * 
   * @return A linha de comando.
   */
  public String getCommandLine() {
    return CommandLineParameterFormatter.formatCommandLine(getName(),
      getValue(), getCommandLinePattern());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandLine(CommandLineContext context) {
    return getCommandLine();
  }

  /**
   * Obtm o padro para a linha de comando.
   * 
   * @return .
   */
  public String getCommandLinePattern() {
    return commandLinePattern;
  }

  /**
   * Obtm o valor do parmetro.
   * 
   * @return O valor do parmetro.
   */
  public String getValue() {
    return this.value;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void importValue(Map<String, Object> values) {
    // No faz nada
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSetDefaultValue() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isVisible() {
    return false;
  }

  /**
   * {@inheritDoc}
   * 
   * No faz nada, pois o parmetro  do tipo hidden que no  altervel.
   */
  @Override
  public void resetValue() {
    // No faz nada
  }

  /**
   * Atribui o padro da linha de comando a este parmetro.
   * 
   * @param commandLinePattern O padro para a linha de comando (No aceita
   *        {@code null}).
   */
  private void setCommandLinePattern(String commandLinePattern) {
    if (commandLinePattern == null) {
      throw new IllegalArgumentException(
        "O parmetro commandLinePattern est nulo.");
    }
    this.commandLinePattern = commandLinePattern;
  }

  /**
   * Atribui um valor a este parmetro.
   * 
   * @param value O valor (No aceita {@code null}).
   */
  private void setValue(String value) {
    if (value == null) {
      throw new IllegalArgumentException("O parmetro value est nulo.");
    }
    this.value = value;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validate(ValidationContext context) {
    return new ValidationSuccess();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setVisible(boolean isVisible) {
    return false;
  }
}
